var __defProp = Object.defineProperty;
var __export = (target, all) => {
  for (var name in all)
    __defProp(target, name, { get: all[name], enumerable: true });
};

// server/index-prod.ts
import fs from "node:fs";
import path from "node:path";
import express2 from "express";

// server/app.ts
import express from "express";

// server/routes.ts
import { createServer } from "http";

// shared/schema.ts
var schema_exports = {};
__export(schema_exports, {
  branches: () => branches,
  branchesRelations: () => branchesRelations,
  dynamicSavingsPlanContributions: () => dynamicSavingsPlanContributions,
  dynamicSavingsPlanContributionsRelations: () => dynamicSavingsPlanContributionsRelations,
  dynamicSavingsPlans: () => dynamicSavingsPlans,
  dynamicSavingsPlansRelations: () => dynamicSavingsPlansRelations,
  insertBranchSchema: () => insertBranchSchema,
  insertDynamicSavingsPlanContributionSchema: () => insertDynamicSavingsPlanContributionSchema,
  insertDynamicSavingsPlanSchema: () => insertDynamicSavingsPlanSchema,
  insertInvestmentTypeSchema: () => insertInvestmentTypeSchema,
  insertLoanSchema: () => insertLoanSchema,
  insertMemberInvestmentSchema: () => insertMemberInvestmentSchema,
  insertMemberSchema: () => insertMemberSchema,
  insertNotificationSchema: () => insertNotificationSchema,
  insertPlanContributionSchema: () => insertPlanContributionSchema,
  insertSavingsPlanSchema: () => insertSavingsPlanSchema,
  insertSavingsPlanTypeSchema: () => insertSavingsPlanTypeSchema,
  insertStaffSchema: () => insertStaffSchema,
  insertTransactionSchema: () => insertTransactionSchema,
  insertYearlyPlanContributionSchema: () => insertYearlyPlanContributionSchema,
  insertYearlySavingsPlanSchema: () => insertYearlySavingsPlanSchema,
  investmentTypes: () => investmentTypes,
  investmentTypesRelations: () => investmentTypesRelations,
  loans: () => loans,
  loansRelations: () => loansRelations,
  loginSchema: () => loginSchema,
  memberInvestments: () => memberInvestments,
  memberInvestmentsRelations: () => memberInvestmentsRelations,
  members: () => members,
  membersRelations: () => membersRelations,
  notifications: () => notifications,
  notificationsRelations: () => notificationsRelations,
  planContributions: () => planContributions,
  planContributionsRelations: () => planContributionsRelations,
  savingsPlanTypes: () => savingsPlanTypes,
  savingsPlanTypesRelations: () => savingsPlanTypesRelations,
  savingsPlans: () => savingsPlans,
  savingsPlansRelations: () => savingsPlansRelations,
  sessions: () => sessions,
  staff: () => staff,
  staffRelations: () => staffRelations,
  transactions: () => transactions,
  transactionsRelations: () => transactionsRelations,
  yearlyPlanContributions: () => yearlyPlanContributions,
  yearlyPlanContributionsRelations: () => yearlyPlanContributionsRelations,
  yearlySavingsPlans: () => yearlySavingsPlans,
  yearlySavingsPlansRelations: () => yearlySavingsPlansRelations
});
import { sql } from "drizzle-orm";
import {
  pgTable,
  serial,
  varchar,
  text,
  integer,
  decimal,
  timestamp,
  boolean
} from "drizzle-orm/pg-core";
import { relations } from "drizzle-orm";
import { createInsertSchema } from "drizzle-zod";
import { z } from "zod";
var branches = pgTable("branches", {
  id: varchar("id").primaryKey().default(sql`gen_random_uuid()`),
  name: text("name").notNull(),
  code: varchar("code", { length: 10 }).unique().notNull(),
  address: text("address"),
  phone: text("phone"),
  status: text("status").notNull().default("active"),
  createdAt: timestamp("created_at").notNull().defaultNow(),
  updatedAt: timestamp("updated_at").notNull().defaultNow()
});
var staff = pgTable("staff", {
  id: varchar("id").primaryKey().default(sql`gen_random_uuid()`),
  branchId: varchar("branch_id").notNull().references(() => branches.id, { onDelete: "cascade" }),
  name: text("name").notNull(),
  email: text("email"),
  phone: text("phone"),
  username: text("username").unique(),
  password: text("password"),
  role: text("role").notNull().default("collector"),
  status: text("status").notNull().default("active"),
  createdAt: timestamp("created_at").notNull().defaultNow(),
  updatedAt: timestamp("updated_at").notNull().defaultNow()
});
var members = pgTable("members", {
  id: varchar("id").primaryKey().default(sql`gen_random_uuid()`),
  staffId: varchar("staff_id").references(() => staff.id, { onDelete: "set null" }),
  name: text("name").notNull(),
  email: text("email"),
  phone: text("phone"),
  address: text("address"),
  joinDate: timestamp("join_date").notNull().defaultNow(),
  status: text("status").notNull().default("active"),
  totalSavings: decimal("total_savings", { precision: 12, scale: 2 }).notNull().default("0"),
  totalPayouts: decimal("total_payouts", { precision: 12, scale: 2 }).notNull().default("0"),
  balance: decimal("balance", { precision: 12, scale: 2 }).notNull().default("0"),
  walletNumber: varchar("wallet_number", { length: 10 }).unique(),
  walletBalance: decimal("wallet_balance", { precision: 12, scale: 2 }).notNull().default("0")
});
var transactions = pgTable("transactions", {
  id: varchar("id").primaryKey().default(sql`gen_random_uuid()`),
  memberId: varchar("member_id").notNull().references(() => members.id, { onDelete: "cascade" }),
  planId: varchar("plan_id").references(() => savingsPlans.id, { onDelete: "set null" }),
  type: text("type").notNull(),
  amount: decimal("amount", { precision: 12, scale: 2 }).notNull(),
  date: timestamp("date").notNull().defaultNow(),
  paymentMethod: text("payment_method"),
  notes: text("notes"),
  status: text("status").notNull().default("completed"),
  payoutDestination: text("payout_destination"),
  payoutAccountNumber: text("payout_account_number"),
  payoutAccountName: text("payout_account_name"),
  payoutBankName: text("payout_bank_name"),
  processedBy: varchar("processed_by").references(() => staff.id, { onDelete: "set null" }),
  processedAt: timestamp("processed_at"),
  createdAt: timestamp("created_at").notNull().defaultNow()
});
var savingsPlanTypes = pgTable("savings_plan_types", {
  id: serial("id").primaryKey(),
  name: text("name").notNull(),
  category: text("category").notNull(),
  // e.g., "food", "education", "investment", "emergency"
  description: text("description"),
  defaultDuration: integer("default_duration").notNull(),
  // in days
  defaultMaxContributions: integer("default_max_contributions").notNull(),
  defaultInterestRate: text("default_interest_rate").notNull(),
  // monthly rate as string
  defaultBreakFee: text("default_break_fee").notNull(),
  // percentage as string
  defaultEarlyWithdrawalPenalty: text("default_early_withdrawal_penalty").notNull(),
  // percentage as string
  isActive: boolean("is_active").notNull().default(true),
  canBreakAfterDays: integer("can_break_after_days").notNull().default(31),
  // minimum days before breaking
  profitCalculationType: text("profit_calculation_type").notNull().default("monthly"),
  // "monthly", "quarterly", "biannually", "yearly"
  createdAt: timestamp("created_at").notNull().default(sql`now()`),
  updatedAt: timestamp("updated_at").notNull().default(sql`now()`)
});
var dynamicSavingsPlans = pgTable("dynamic_savings_plans", {
  id: serial("id").primaryKey(),
  planTypeId: integer("plan_type_id").notNull(),
  memberId: text("member_id").notNull(),
  planName: text("plan_name").notNull(),
  targetAmount: text("target_amount").notNull(),
  contributionAmount: text("contribution_amount").notNull(),
  maxContributions: integer("max_contributions").notNull(),
  currentContributions: integer("current_contributions").notNull().default(0),
  totalSaved: text("total_saved").notNull().default("0"),
  interestRate: text("interest_rate").notNull(),
  // monthly rate
  breakFee: text("break_fee").notNull(),
  // percentage
  earlyWithdrawalPenalty: text("early_withdrawal_penalty").notNull(),
  // percentage
  status: text("status").notNull().default("active"),
  // "active", "matured", "broken", "completed"
  startDate: timestamp("start_date").notNull().default(sql`now()`),
  maturityDate: timestamp("maturity_date").notNull(),
  completedDate: timestamp("completed_date"),
  profitEarned: text("profit_earned").notNull().default("0"),
  totalWithProfit: text("total_with_profit").notNull().default("0"),
  payoutStatus: text("payout_status").notNull().default("none"),
  // "none", "pending", "completed"
  canBreakAfterDays: integer("can_break_after_days").notNull(),
  profitCalculationType: text("profit_calculation_type").notNull().default("monthly"),
  createdAt: timestamp("created_at").notNull().default(sql`now()`),
  updatedAt: timestamp("updated_at").notNull().default(sql`now()`)
});
var dynamicSavingsPlanContributions = pgTable("dynamic_savings_plan_contributions", {
  id: serial("id").primaryKey(),
  planId: integer("plan_id").notNull(),
  memberId: text("member_id").notNull(),
  amount: text("amount").notNull(),
  date: timestamp("date").notNull().default(sql`now()`),
  paymentMethod: text("payment_method"),
  notes: text("notes"),
  contributionNumber: integer("contribution_number").notNull(),
  recordedBy: text("recorded_by"),
  createdAt: timestamp("created_at").notNull().default(sql`now()`)
});
var savingsPlans = pgTable("savings_plans", {
  id: varchar("id").primaryKey().default(sql`gen_random_uuid()`),
  memberId: varchar("member_id").notNull().references(() => members.id, { onDelete: "cascade" }),
  planName: text("plan_name").notNull(),
  targetAmount: decimal("target_amount", { precision: 12, scale: 2 }).notNull(),
  contributionAmount: decimal("contribution_amount", { precision: 12, scale: 2 }).notNull(),
  status: text("status").notNull().default("active"),
  contributionsCount: integer("contributions_count").notNull().default(0),
  maxContributions: integer("max_contributions").notNull().default(31),
  maxDays: integer("max_days").notNull().default(62),
  startDate: timestamp("start_date").notNull().defaultNow(),
  completedDate: timestamp("completed_date"),
  totalSaved: decimal("total_saved", { precision: 12, scale: 2 }).notNull().default("0"),
  payoutStatus: text("payout_status").default("none"),
  createdAt: timestamp("created_at").notNull().defaultNow()
});
var planContributions = pgTable("plan_contributions", {
  id: varchar("id").primaryKey().default(sql`gen_random_uuid()`),
  planId: varchar("plan_id").notNull().references(() => savingsPlans.id, { onDelete: "cascade" }),
  memberId: varchar("member_id").notNull().references(() => members.id, { onDelete: "cascade" }),
  amount: decimal("amount", { precision: 12, scale: 2 }).notNull(),
  contributionNumber: integer("contribution_number").notNull(),
  date: timestamp("date").notNull().defaultNow(),
  paymentMethod: text("payment_method"),
  notes: text("notes"),
  recordedBy: varchar("recorded_by").references(() => staff.id, { onDelete: "set null" }),
  createdAt: timestamp("created_at").notNull().defaultNow()
});
var yearlySavingsPlans = pgTable("yearly_savings_plans", {
  id: varchar("id").primaryKey().default(sql`gen_random_uuid()`),
  memberId: varchar("member_id").notNull().references(() => members.id, { onDelete: "cascade" }),
  planName: text("plan_name").notNull(),
  targetAmount: decimal("target_amount", { precision: 12, scale: 2 }).notNull(),
  contributionAmount: decimal("contribution_amount", { precision: 12, scale: 2 }).notNull(),
  status: text("status").notNull().default("active"),
  contributionsCount: integer("contributions_count").notNull().default(0),
  maxContributions: integer("max_contributions").notNull().default(372),
  maxDays: integer("max_days").notNull().default(372),
  startDate: timestamp("start_date").notNull().defaultNow(),
  maturityDate: timestamp("maturity_date").notNull(),
  completedDate: timestamp("completed_date"),
  totalSaved: decimal("total_saved", { precision: 12, scale: 2 }).notNull().default("0"),
  profitRate: decimal("profit_rate", { precision: 5, scale: 2 }).notNull().default("5.00"),
  profitEarned: decimal("profit_earned", { precision: 12, scale: 2 }).notNull().default("0"),
  payoutStatus: text("payout_status").default("none"),
  createdAt: timestamp("created_at").notNull().defaultNow()
});
var yearlyPlanContributions = pgTable("yearly_plan_contributions", {
  id: varchar("id").primaryKey().default(sql`gen_random_uuid()`),
  planId: varchar("plan_id").notNull().references(() => yearlySavingsPlans.id, { onDelete: "cascade" }),
  memberId: varchar("member_id").notNull().references(() => members.id, { onDelete: "cascade" }),
  amount: decimal("amount", { precision: 12, scale: 2 }).notNull(),
  contributionNumber: integer("contribution_number").notNull(),
  date: timestamp("date").notNull().defaultNow(),
  paymentMethod: text("payment_method"),
  notes: text("notes"),
  recordedBy: varchar("recorded_by").references(() => staff.id, { onDelete: "set null" }),
  createdAt: timestamp("created_at").notNull().defaultNow()
});
var notifications = pgTable("notifications", {
  id: varchar("id").primaryKey().default(sql`gen_random_uuid()`),
  type: text("type").notNull(),
  title: text("title").notNull(),
  message: text("message").notNull(),
  memberId: varchar("member_id").references(() => members.id, { onDelete: "cascade" }),
  transactionId: varchar("transaction_id").references(() => transactions.id, { onDelete: "cascade" }),
  read: text("read").notNull().default("false"),
  createdAt: timestamp("created_at").notNull().defaultNow()
});
var sessions = pgTable("sessions", {
  sid: varchar("sid").primaryKey(),
  sess: text("sess").notNull(),
  expire: timestamp("expire").notNull()
});
var loans = pgTable("loans", {
  id: varchar("id").primaryKey().default(sql`gen_random_uuid()`),
  memberId: varchar("member_id").notNull().references(() => members.id, { onDelete: "cascade" }),
  planId: varchar("plan_id").notNull().references(() => savingsPlans.id, { onDelete: "cascade" }),
  amount: decimal("amount", { precision: 12, scale: 2 }).notNull(),
  interestRate: decimal("interest_rate", { precision: 5, scale: 2 }).notNull().default("5.00"),
  totalRepayment: decimal("total_repayment", { precision: 12, scale: 2 }).notNull(),
  amountPaid: decimal("amount_paid", { precision: 12, scale: 2 }).notNull().default("0"),
  status: text("status").notNull().default("pending"),
  requestedAt: timestamp("requested_at").notNull().defaultNow(),
  approvedAt: timestamp("approved_at"),
  approvedBy: varchar("approved_by").references(() => staff.id, { onDelete: "set null" }),
  dueDate: timestamp("due_date"),
  notes: text("notes")
});
var branchesRelations = relations(branches, ({ many }) => ({
  staff: many(staff)
}));
var savingsPlanTypesRelations = relations(savingsPlanTypes, ({ many }) => ({
  dynamicPlans: many(dynamicSavingsPlans)
}));
var dynamicSavingsPlansRelations = relations(dynamicSavingsPlans, ({ one, many }) => ({
  planType: one(savingsPlanTypes, {
    fields: [dynamicSavingsPlans.planTypeId],
    references: [savingsPlanTypes.id]
  }),
  member: one(members, {
    fields: [dynamicSavingsPlans.memberId],
    references: [members.id]
  }),
  contributions: many(dynamicSavingsPlanContributions)
}));
var dynamicSavingsPlanContributionsRelations = relations(dynamicSavingsPlanContributions, ({ one }) => ({
  plan: one(dynamicSavingsPlans, {
    fields: [dynamicSavingsPlanContributions.planId],
    references: [dynamicSavingsPlans.id]
  }),
  member: one(members, {
    fields: [dynamicSavingsPlanContributions.memberId],
    references: [members.id]
  })
}));
var staffRelations = relations(staff, ({ one, many }) => ({
  branch: one(branches, {
    fields: [staff.branchId],
    references: [branches.id]
  }),
  members: many(members)
}));
var membersRelations = relations(members, ({ one, many }) => ({
  staff: one(staff, {
    fields: [members.staffId],
    references: [staff.id]
  }),
  transactions: many(transactions),
  savingsPlans: many(savingsPlans),
  yearlySavingsPlans: many(yearlySavingsPlans),
  yearlyPlanContributions: many(yearlyPlanContributions),
  dynamicSavingsPlans: many(dynamicSavingsPlans),
  dynamicSavingsPlanContributions: many(dynamicSavingsPlanContributions),
  loans: many(loans),
  notifications: many(notifications)
}));
var transactionsRelations = relations(transactions, ({ one }) => ({
  member: one(members, {
    fields: [transactions.memberId],
    references: [members.id]
  }),
  plan: one(savingsPlans, {
    fields: [transactions.planId],
    references: [savingsPlans.id]
  }),
  processedByStaff: one(staff, {
    fields: [transactions.processedBy],
    references: [staff.id]
  })
}));
var savingsPlansRelations = relations(savingsPlans, ({ one, many }) => ({
  member: one(members, {
    fields: [savingsPlans.memberId],
    references: [members.id]
  }),
  contributions: many(planContributions)
}));
var yearlySavingsPlansRelations = relations(yearlySavingsPlans, ({ one, many }) => ({
  member: one(members, {
    fields: [yearlySavingsPlans.memberId],
    references: [members.id]
  }),
  contributions: many(yearlyPlanContributions)
}));
var planContributionsRelations = relations(planContributions, ({ one }) => ({
  plan: one(savingsPlans, {
    fields: [planContributions.planId],
    references: [savingsPlans.id]
  }),
  member: one(members, {
    fields: [planContributions.memberId],
    references: [members.id]
  }),
  recordedByStaff: one(staff, {
    fields: [planContributions.recordedBy],
    references: [staff.id]
  })
}));
var yearlyPlanContributionsRelations = relations(yearlyPlanContributions, ({ one }) => ({
  plan: one(yearlySavingsPlans, {
    fields: [yearlyPlanContributions.planId],
    references: [yearlySavingsPlans.id]
  }),
  member: one(members, {
    fields: [yearlyPlanContributions.memberId],
    references: [members.id]
  }),
  recordedByStaff: one(staff, {
    fields: [yearlyPlanContributions.recordedBy],
    references: [staff.id]
  })
}));
var notificationsRelations = relations(notifications, ({ one }) => ({
  member: one(members, {
    fields: [notifications.memberId],
    references: [members.id]
  }),
  transaction: one(transactions, {
    fields: [notifications.transactionId],
    references: [transactions.id]
  })
}));
var loansRelations = relations(loans, ({ one }) => ({
  member: one(members, {
    fields: [loans.memberId],
    references: [members.id]
  }),
  plan: one(savingsPlans, {
    fields: [loans.planId],
    references: [savingsPlans.id]
  }),
  approvedByStaff: one(staff, {
    fields: [loans.approvedBy],
    references: [staff.id]
  })
}));
var investmentTypes = pgTable("investment_types", {
  id: varchar("id").primaryKey().default(sql`gen_random_uuid()`),
  name: text("name").notNull(),
  category: text("category").notNull(),
  description: text("description"),
  minimumDeposit: decimal("minimum_deposit", { precision: 12, scale: 2 }).notNull(),
  interestRate: decimal("interest_rate", { precision: 5, scale: 2 }).notNull(),
  paymentPlan: text("payment_plan").notNull(),
  durationDays: integer("duration_days").notNull(),
  breakFee: decimal("break_fee", { precision: 5, scale: 2 }).notNull().default("0"),
  isBreakable: boolean("is_breakable").notNull().default(true),
  status: text("status").notNull().default("active"),
  createdBy: varchar("created_by").references(() => staff.id, { onDelete: "set null" }),
  createdAt: timestamp("created_at").notNull().defaultNow(),
  updatedAt: timestamp("updated_at").notNull().defaultNow()
});
var memberInvestments = pgTable("member_investments", {
  id: varchar("id").primaryKey().default(sql`gen_random_uuid()`),
  memberId: varchar("member_id").notNull().references(() => members.id, { onDelete: "cascade" }),
  investmentTypeId: varchar("investment_type_id").notNull().references(() => investmentTypes.id, { onDelete: "cascade" }),
  amount: decimal("amount", { precision: 12, scale: 2 }).notNull(),
  interestRate: decimal("interest_rate", { precision: 5, scale: 2 }).notNull(),
  expectedReturn: decimal("expected_return", { precision: 12, scale: 2 }).notNull(),
  startDate: timestamp("start_date").notNull().defaultNow(),
  maturityDate: timestamp("maturity_date").notNull(),
  status: text("status").notNull().default("active"),
  breakFeeApplied: decimal("break_fee_applied", { precision: 12, scale: 2 }),
  actualReturn: decimal("actual_return", { precision: 12, scale: 2 }),
  completedDate: timestamp("completed_date"),
  notes: text("notes"),
  assignedBy: varchar("assigned_by").references(() => staff.id, { onDelete: "set null" }),
  createdAt: timestamp("created_at").notNull().defaultNow()
});
var investmentTypesRelations = relations(investmentTypes, ({ one, many }) => ({
  createdByStaff: one(staff, {
    fields: [investmentTypes.createdBy],
    references: [staff.id]
  }),
  memberInvestments: many(memberInvestments)
}));
var memberInvestmentsRelations = relations(memberInvestments, ({ one }) => ({
  member: one(members, {
    fields: [memberInvestments.memberId],
    references: [members.id]
  }),
  investmentType: one(investmentTypes, {
    fields: [memberInvestments.investmentTypeId],
    references: [investmentTypes.id]
  }),
  assignedByStaff: one(staff, {
    fields: [memberInvestments.assignedBy],
    references: [staff.id]
  })
}));
var insertBranchSchema = createInsertSchema(branches).omit({
  id: true,
  createdAt: true,
  updatedAt: true
}).extend({
  name: z.string().min(1, "Branch name is required"),
  code: z.string().min(2, "Branch code is required").max(10, "Branch code must be 10 characters or less"),
  address: z.string().optional(),
  phone: z.string().optional(),
  status: z.enum(["active", "inactive"]).default("active")
});
var insertStaffSchema = createInsertSchema(staff).omit({
  id: true,
  createdAt: true,
  updatedAt: true
}).extend({
  branchId: z.string().min(1, "Branch is required"),
  name: z.string().min(1, "Staff name is required"),
  email: z.string().email().optional().or(z.literal("")),
  phone: z.string().optional(),
  username: z.string().min(3, "Username must be at least 3 characters").optional(),
  password: z.string().min(6, "Password must be at least 6 characters").optional(),
  role: z.enum(["manager", "collector", "admin"]).default("collector"),
  status: z.enum(["active", "inactive"]).default("active")
});
var insertMemberSchema = createInsertSchema(members).omit({
  id: true,
  joinDate: true,
  totalSavings: true,
  totalPayouts: true,
  balance: true,
  walletNumber: true,
  walletBalance: true
}).extend({
  staffId: z.string().optional().nullable(),
  name: z.string().min(1, "Name is required"),
  email: z.string().email().optional().or(z.literal("")),
  phone: z.string().optional(),
  address: z.string().optional(),
  status: z.enum(["active", "inactive"]).default("active")
});
var insertTransactionSchema = createInsertSchema(transactions).omit({
  id: true,
  createdAt: true,
  date: true,
  processedAt: true
}).extend({
  memberId: z.string().min(1, "Member is required"),
  planId: z.string().optional().nullable(),
  type: z.enum(["savings", "payout"]),
  amount: z.string().min(1, "Amount is required").refine((val) => !isNaN(Number(val)) && Number(val) > 0, {
    message: "Amount must be a positive number"
  }),
  paymentMethod: z.string().optional(),
  notes: z.string().optional(),
  status: z.enum(["completed", "pending", "approved", "rejected"]).default("completed"),
  date: z.string().optional(),
  payoutDestination: z.string().optional(),
  payoutAccountNumber: z.string().optional(),
  payoutAccountName: z.string().optional(),
  payoutBankName: z.string().optional(),
  processedBy: z.string().optional()
});
var insertSavingsPlanTypeSchema = createInsertSchema(savingsPlanTypes).omit({
  id: true,
  createdAt: true,
  updatedAt: true
}).extend({
  name: z.string().min(1, "Plan type name is required"),
  category: z.string().min(1, "Category is required"),
  defaultDuration: z.number().min(1, "Duration must be at least 1 day"),
  defaultMaxContributions: z.number().min(1, "Max contributions must be at least 1"),
  defaultInterestRate: z.string().min(1, "Interest rate is required"),
  defaultBreakFee: z.string().min(1, "Break fee is required"),
  defaultEarlyWithdrawalPenalty: z.string().min(1, "Early withdrawal penalty is required"),
  profitCalculationType: z.enum(["monthly", "quarterly", "biannually", "yearly"]).default("monthly")
});
var insertDynamicSavingsPlanSchema = createInsertSchema(dynamicSavingsPlans).omit({
  id: true,
  currentContributions: true,
  totalSaved: true,
  status: true,
  startDate: true,
  completedDate: true,
  profitEarned: true,
  totalWithProfit: true,
  payoutStatus: true,
  createdAt: true,
  updatedAt: true
}).extend({
  planTypeId: z.number().min(1, "Plan type is required"),
  memberId: z.string().min(1, "Member is required"),
  planName: z.string().min(1, "Plan name is required"),
  targetAmount: z.string().min(1, "Target amount is required"),
  contributionAmount: z.string().min(1, "Contribution amount is required"),
  maxContributions: z.number().min(1, "Max contributions must be at least 1"),
  interestRate: z.string().min(1, "Interest rate is required"),
  breakFee: z.string().min(1, "Break fee is required"),
  earlyWithdrawalPenalty: z.string().min(1, "Early withdrawal penalty is required"),
  maturityDate: z.string().min(1, "Maturity date is required"),
  canBreakAfterDays: z.number().min(1, "Can break after days must be at least 1"),
  profitCalculationType: z.enum(["monthly", "quarterly", "biannually", "yearly"]).default("monthly")
});
var insertDynamicSavingsPlanContributionSchema = createInsertSchema(dynamicSavingsPlanContributions).omit({
  id: true,
  contributionNumber: true,
  createdAt: true
}).extend({
  planId: z.number().min(1, "Plan is required"),
  memberId: z.string().min(1, "Member is required"),
  amount: z.string().min(1, "Amount is required"),
  date: z.string().optional(),
  paymentMethod: z.string().optional(),
  notes: z.string().optional(),
  recordedBy: z.string().optional()
});
var insertSavingsPlanSchema = createInsertSchema(savingsPlans).omit({
  id: true,
  contributionsCount: true,
  startDate: true,
  completedDate: true,
  totalSaved: true,
  createdAt: true,
  status: true,
  payoutStatus: true
}).extend({
  memberId: z.string().min(1, "Member is required"),
  planName: z.string().min(1, "Plan name is required"),
  targetAmount: z.string().min(1, "Target amount is required").refine((val) => !isNaN(Number(val)) && Number(val) > 0, {
    message: "Target amount must be a positive number"
  }),
  contributionAmount: z.string().min(1, "Contribution amount is required").refine((val) => !isNaN(Number(val)) && Number(val) > 0, {
    message: "Contribution amount must be a positive number"
  }),
  maxContributions: z.number().int().positive().default(31),
  maxDays: z.number().int().positive().default(62)
});
var insertPlanContributionSchema = createInsertSchema(planContributions).omit({
  id: true,
  createdAt: true,
  date: true,
  contributionNumber: true
}).extend({
  planId: z.string().min(1, "Plan ID is required"),
  memberId: z.string().min(1, "Member ID is required"),
  amount: z.string().min(1, "Amount is required").refine((val) => !isNaN(Number(val)) && Number(val) > 0, {
    message: "Amount must be a positive number"
  }),
  paymentMethod: z.string().optional(),
  notes: z.string().optional(),
  recordedBy: z.string().optional()
});
var insertYearlySavingsPlanSchema = createInsertSchema(yearlySavingsPlans).omit({
  id: true,
  contributionsCount: true,
  startDate: true,
  completedDate: true,
  totalSaved: true,
  profitEarned: true,
  createdAt: true,
  status: true,
  payoutStatus: true
}).extend({
  memberId: z.string().min(1, "Member is required"),
  planName: z.string().min(1, "Plan name is required"),
  targetAmount: z.string().min(1, "Target amount is required").refine((val) => !isNaN(Number(val)) && Number(val) > 0, {
    message: "Target amount must be a positive number"
  }),
  contributionAmount: z.string().min(1, "Contribution amount is required").refine((val) => !isNaN(Number(val)) && Number(val) > 0, {
    message: "Contribution amount must be a positive number"
  }),
  maxContributions: z.number().int().positive().default(372),
  maxDays: z.number().int().positive().default(372),
  profitRate: z.string().refine((val) => !isNaN(Number(val)) && Number(val) >= 0, {
    message: "Profit rate must be zero or positive"
  }).default("5.00"),
  maturityDate: z.string().min(1, "Maturity date is required")
});
var insertYearlyPlanContributionSchema = createInsertSchema(yearlyPlanContributions).omit({
  id: true,
  createdAt: true,
  date: true,
  contributionNumber: true
}).extend({
  planId: z.string().min(1, "Plan ID is required"),
  memberId: z.string().min(1, "Member ID is required"),
  amount: z.string().min(1, "Amount is required").refine((val) => !isNaN(Number(val)) && Number(val) > 0, {
    message: "Amount must be a positive number"
  }),
  paymentMethod: z.string().optional(),
  notes: z.string().optional(),
  recordedBy: z.string().optional()
});
var insertNotificationSchema = createInsertSchema(notifications).omit({
  id: true,
  createdAt: true
}).extend({
  type: z.enum(["pending_payout", "payout_approved", "plan_completed", "reminder", "plan_closed", "loan_requested", "loan_approved", "loan_rejected", "yearly_plan_matured", "monthly_payout"]),
  title: z.string().min(1, "Title is required"),
  message: z.string().min(1, "Message is required"),
  memberId: z.string().optional(),
  transactionId: z.string().optional(),
  read: z.enum(["true", "false"]).default("false")
});
var insertLoanSchema = createInsertSchema(loans).omit({
  id: true,
  requestedAt: true,
  approvedAt: true,
  amountPaid: true,
  status: true,
  interestRate: true,
  totalRepayment: true,
  dueDate: true,
  approvedBy: true
}).extend({
  memberId: z.string().min(1, "Member is required"),
  planId: z.string().min(1, "Savings plan is required"),
  amount: z.string().min(1, "Amount is required").refine((val) => !isNaN(Number(val)) && Number(val) > 0, {
    message: "Amount must be a positive number"
  }),
  notes: z.string().optional()
});
var insertInvestmentTypeSchema = createInsertSchema(investmentTypes).omit({
  id: true,
  createdAt: true,
  updatedAt: true
}).extend({
  name: z.string().min(1, "Investment name is required"),
  category: z.string().min(1, "Category is required"),
  description: z.string().optional(),
  minimumDeposit: z.string().min(1, "Minimum deposit is required").refine((val) => !isNaN(Number(val)) && Number(val) > 0, {
    message: "Minimum deposit must be a positive number"
  }),
  interestRate: z.string().min(1, "Interest rate is required").refine((val) => !isNaN(Number(val)) && Number(val) >= 0, {
    message: "Interest rate must be zero or positive"
  }),
  paymentPlan: z.enum(["daily", "weekly", "monthly", "quarterly", "yearly", "maturity"]),
  durationDays: z.number().int().positive("Duration must be positive"),
  breakFee: z.string().refine((val) => !isNaN(Number(val)) && Number(val) >= 0, {
    message: "Break fee must be zero or positive"
  }).default("0"),
  isBreakable: z.boolean().default(true),
  status: z.enum(["active", "inactive"]).default("active"),
  createdBy: z.string().optional()
});
var insertMemberInvestmentSchema = createInsertSchema(memberInvestments).omit({
  id: true,
  createdAt: true,
  startDate: true,
  expectedReturn: true,
  maturityDate: true,
  breakFeeApplied: true,
  actualReturn: true,
  completedDate: true
}).extend({
  memberId: z.string().min(1, "Member is required"),
  investmentTypeId: z.string().min(1, "Investment type is required"),
  amount: z.string().min(1, "Amount is required").refine((val) => !isNaN(Number(val)) && Number(val) > 0, {
    message: "Amount must be a positive number"
  }),
  interestRate: z.string().optional(),
  status: z.enum(["active", "matured", "broken", "completed"]).default("active"),
  notes: z.string().optional(),
  assignedBy: z.string().optional()
});
var loginSchema = z.object({
  username: z.string().min(1, "Username is required"),
  password: z.string().min(1, "Password is required")
});

// server/db.ts
import { Pool, neonConfig } from "@neondatabase/serverless";
import { drizzle } from "drizzle-orm/neon-serverless";
import ws from "ws";
neonConfig.webSocketConstructor = ws;
if (!process.env.DATABASE_URL) {
  throw new Error(
    "DATABASE_URL must be set. Did you forget to provision a database?"
  );
}
var pool = new Pool({ connectionString: process.env.DATABASE_URL });
var db = drizzle({ client: pool, schema: schema_exports });

// server/storage.ts
import { eq as eq2, desc, sql as sql2, lt, and as and2 } from "drizzle-orm";

// server/auth.ts
import passport from "passport";
import { Strategy as LocalStrategy } from "passport-local";
import session from "express-session";
import connectPg from "connect-pg-simple";
import { scrypt, randomBytes, timingSafeEqual } from "crypto";
import { promisify } from "util";
import { eq, and } from "drizzle-orm";
var scryptAsync = promisify(scrypt);
var PostgresSessionStore = connectPg(session);
async function hashPassword(password) {
  const salt = randomBytes(16).toString("hex");
  const buf = await scryptAsync(password, salt, 64);
  return `${buf.toString("hex")}.${salt}`;
}
async function comparePasswords(supplied, stored) {
  const [hashed, salt] = stored.split(".");
  const hashedBuf = Buffer.from(hashed, "hex");
  const suppliedBuf = await scryptAsync(supplied, salt, 64);
  return timingSafeEqual(hashedBuf, suppliedBuf);
}
async function getStaffByUsername(username) {
  return db.select().from(staff).where(
    and(
      eq(staff.username, username),
      eq(staff.status, "active")
    )
  ).limit(1);
}
function setupAuth(app2) {
  const store = new PostgresSessionStore({ pool, createTableIfMissing: true });
  const sessionSettings = {
    secret: process.env.SESSION_SECRET || "apexl-savings-secret-key-2024",
    resave: false,
    saveUninitialized: false,
    store,
    cookie: {
      maxAge: 24 * 60 * 60 * 1e3,
      httpOnly: true,
      secure: false,
      sameSite: "lax"
    }
  };
  app2.set("trust proxy", 1);
  app2.use(session(sessionSettings));
  app2.use(passport.initialize());
  app2.use(passport.session());
  passport.use(
    new LocalStrategy(async (username, password, done) => {
      try {
        const [user] = await getStaffByUsername(username);
        if (!user || !user.password) {
          return done(null, false, { message: "Invalid username or password" });
        }
        const isValid = await comparePasswords(password, user.password);
        if (!isValid) {
          return done(null, false, { message: "Invalid username or password" });
        }
        return done(null, user);
      } catch (error) {
        return done(error);
      }
    })
  );
  passport.serializeUser((user, done) => done(null, user.id));
  passport.deserializeUser(async (id, done) => {
    try {
      const [user] = await db.select().from(staff).where(eq(staff.id, id)).limit(1);
      done(null, user || null);
    } catch (error) {
      done(error);
    }
  });
  app2.post("/api/login", (req, res, next) => {
    passport.authenticate("local", (err, user, info) => {
      if (err) {
        return res.status(500).json({ message: "Authentication error" });
      }
      if (!user) {
        return res.status(401).json({ message: info?.message || "Invalid credentials" });
      }
      req.login(user, (loginErr) => {
        if (loginErr) {
          return res.status(500).json({ message: "Login error" });
        }
        const { password: _, ...userWithoutPassword } = user;
        return res.status(200).json(userWithoutPassword);
      });
    })(req, res, next);
  });
  app2.post("/api/logout", (req, res, next) => {
    req.logout((err) => {
      if (err) return next(err);
      res.sendStatus(200);
    });
  });
  app2.get("/api/user", (req, res) => {
    if (!req.isAuthenticated()) return res.sendStatus(401);
    const { password: _, ...userWithoutPassword } = req.user;
    res.json(userWithoutPassword);
  });
}
function requireAuth(req, res, next) {
  if (!req.isAuthenticated()) {
    return res.status(401).json({ message: "Unauthorized" });
  }
  next();
}
function requireRole(...roles) {
  return (req, res, next) => {
    if (!req.isAuthenticated()) {
      return res.status(401).json({ message: "Unauthorized" });
    }
    if (!roles.includes(req.user.role)) {
      return res.status(403).json({ message: "Forbidden: Insufficient permissions" });
    }
    next();
  };
}
function filterByUserBranch(req) {
  const user = req.user;
  if (!user || user.role === "admin") {
    return void 0;
  }
  return user.branchId;
}

// server/storage.ts
var DatabaseStorage = class {
  async getBranch(id) {
    const result = await db.query.branches.findFirst({
      where: eq2(branches.id, id),
      with: {
        staff: true
      }
    });
    return result;
  }
  async getAllBranches() {
    return await db.select().from(branches).orderBy(desc(branches.createdAt));
  }
  async getBranchesPaginated(page, limit) {
    const offset = (page - 1) * limit;
    const [data, countResult] = await Promise.all([
      db.select().from(branches).orderBy(desc(branches.createdAt)).limit(limit).offset(offset),
      db.select({ count: sql2`count(*)` }).from(branches)
    ]);
    const total = Number(countResult[0]?.count) || 0;
    return {
      data,
      total,
      page,
      limit,
      totalPages: Math.ceil(total / limit)
    };
  }
  async createBranch(insertBranch) {
    const [branch] = await db.insert(branches).values(insertBranch).returning();
    return branch;
  }
  async updateBranch(id, insertBranch) {
    const [branch] = await db.update(branches).set({ ...insertBranch, updatedAt: /* @__PURE__ */ new Date() }).where(eq2(branches.id, id)).returning();
    return branch;
  }
  async deleteBranch(id) {
    await db.delete(branches).where(eq2(branches.id, id));
  }
  async getStaff(id) {
    const result = await db.query.staff.findFirst({
      where: eq2(staff.id, id),
      with: {
        branch: true
      }
    });
    return result;
  }
  async getAllStaff() {
    const result = await db.query.staff.findMany({
      with: {
        branch: true
      },
      orderBy: [desc(staff.createdAt)]
    });
    return result;
  }
  async getStaffPaginated(page, limit) {
    const offset = (page - 1) * limit;
    const [data, countResult] = await Promise.all([
      db.query.staff.findMany({
        with: { branch: true },
        orderBy: [desc(staff.createdAt)],
        limit,
        offset
      }),
      db.select({ count: sql2`count(*)` }).from(staff)
    ]);
    const total = Number(countResult[0]?.count) || 0;
    return {
      data,
      total,
      page,
      limit,
      totalPages: Math.ceil(total / limit)
    };
  }
  async getStaffByBranch(branchId) {
    return await db.select().from(staff).where(eq2(staff.branchId, branchId)).orderBy(desc(staff.createdAt));
  }
  async createStaff(insertStaff) {
    let hashedPassword;
    if (insertStaff.password) {
      hashedPassword = await hashPassword(insertStaff.password);
    }
    const [newStaff] = await db.insert(staff).values({
      ...insertStaff,
      password: hashedPassword
    }).returning();
    return newStaff;
  }
  async updateStaff(id, insertStaff) {
    let updateData = { ...insertStaff, updatedAt: /* @__PURE__ */ new Date() };
    if (insertStaff.password) {
      updateData.password = await hashPassword(insertStaff.password);
    }
    const [updatedStaff] = await db.update(staff).set(updateData).where(eq2(staff.id, id)).returning();
    return updatedStaff;
  }
  async deleteStaff(id) {
    await db.delete(staff).where(eq2(staff.id, id));
  }
  async getMember(id) {
    const result = await db.query.members.findFirst({
      where: eq2(members.id, id),
      with: {
        transactions: {
          orderBy: [desc(transactions.date)]
        }
      }
    });
    return result;
  }
  async getMemberWithStaff(id) {
    const result = await db.query.members.findFirst({
      where: eq2(members.id, id),
      with: {
        staff: {
          with: {
            branch: true
          }
        }
      }
    });
    return result;
  }
  async getAllMembers() {
    return await db.select().from(members).orderBy(desc(members.joinDate));
  }
  async getMembersPaginated(page, limit) {
    const offset = (page - 1) * limit;
    const [data, countResult] = await Promise.all([
      db.select().from(members).orderBy(desc(members.joinDate)).limit(limit).offset(offset),
      db.select({ count: sql2`count(*)` }).from(members)
    ]);
    const total = Number(countResult[0]?.count) || 0;
    return {
      data,
      total,
      page,
      limit,
      totalPages: Math.ceil(total / limit)
    };
  }
  async getMembersByStaff(staffId) {
    return await db.select().from(members).where(eq2(members.staffId, staffId)).orderBy(desc(members.joinDate));
  }
  async createMember(insertMember) {
    const walletNumber = Math.floor(Math.random() * 1e10).toString().padStart(10, "0");
    const [member] = await db.insert(members).values({
      ...insertMember,
      walletNumber
    }).returning();
    return member;
  }
  async updateMember(id, insertMember) {
    const [member] = await db.update(members).set(insertMember).where(eq2(members.id, id)).returning();
    return member;
  }
  async deleteMember(id) {
    await db.delete(members).where(eq2(members.id, id));
  }
  async getTransaction(id) {
    const result = await db.query.transactions.findFirst({
      where: eq2(transactions.id, id),
      with: {
        member: true
      }
    });
    return result;
  }
  async getAllTransactions() {
    const result = await db.query.transactions.findMany({
      with: {
        member: true
      },
      orderBy: [desc(transactions.date)]
    });
    return result;
  }
  async getTransactionsPaginated(page, limit) {
    const offset = (page - 1) * limit;
    const [data, countResult] = await Promise.all([
      db.query.transactions.findMany({
        with: { member: true },
        orderBy: [desc(transactions.date)],
        limit,
        offset
      }),
      db.select({ count: sql2`count(*)` }).from(transactions)
    ]);
    const total = Number(countResult[0]?.count) || 0;
    return {
      data,
      total,
      page,
      limit,
      totalPages: Math.ceil(total / limit)
    };
  }
  async getRecentTransactions(limit) {
    const result = await db.query.transactions.findMany({
      with: {
        member: true
      },
      orderBy: [desc(transactions.date)],
      limit
    });
    return result;
  }
  async getTransactionsByType(type) {
    const result = await db.query.transactions.findMany({
      where: eq2(transactions.type, type),
      with: {
        member: true
      },
      orderBy: [desc(transactions.date)]
    });
    return result;
  }
  async getPayoutsPaginated(page, limit) {
    const offset = (page - 1) * limit;
    const [data, countResult] = await Promise.all([
      db.query.transactions.findMany({
        where: eq2(transactions.type, "payout"),
        with: { member: true },
        orderBy: [desc(transactions.date)],
        limit,
        offset
      }),
      db.select({ count: sql2`count(*)` }).from(transactions).where(eq2(transactions.type, "payout"))
    ]);
    const total = Number(countResult[0]?.count) || 0;
    return {
      data,
      total,
      page,
      limit,
      totalPages: Math.ceil(total / limit)
    };
  }
  async createTransaction(insertTransaction) {
    const amount = parseFloat(insertTransaction.amount);
    const memberId = insertTransaction.memberId;
    const type = insertTransaction.type;
    const [transaction] = await db.insert(transactions).values({
      ...insertTransaction,
      date: insertTransaction.date ? new Date(insertTransaction.date) : /* @__PURE__ */ new Date()
    }).returning();
    if (type === "savings") {
      await db.update(members).set({
        totalSavings: sql2`${members.totalSavings} + ${amount}`,
        balance: sql2`${members.balance} + ${amount}`
      }).where(eq2(members.id, memberId));
    } else if (type === "payout" && insertTransaction.status === "completed") {
      await db.update(members).set({
        totalPayouts: sql2`${members.totalPayouts} + ${amount}`,
        balance: sql2`${members.balance} - ${amount}`
      }).where(eq2(members.id, memberId));
    }
    return transaction;
  }
  async updateTransactionStatus(id, status) {
    const transaction = await db.query.transactions.findFirst({
      where: eq2(transactions.id, id)
    });
    if (!transaction) {
      throw new Error("Transaction not found");
    }
    const [updatedTransaction] = await db.update(transactions).set({ status }).where(eq2(transactions.id, id)).returning();
    if (transaction.type === "payout" && status === "completed" && transaction.status !== "completed") {
      const amount = parseFloat(transaction.amount);
      await db.update(members).set({
        totalPayouts: sql2`${members.totalPayouts} + ${amount}`,
        balance: sql2`${members.balance} - ${amount}`
      }).where(eq2(members.id, transaction.memberId));
    }
    return updatedTransaction;
  }
  async completePayoutTransaction(id, payoutDetails) {
    const transaction = await db.query.transactions.findFirst({
      where: eq2(transactions.id, id)
    });
    if (!transaction) {
      throw new Error("Transaction not found");
    }
    if (transaction.type !== "payout") {
      throw new Error("Transaction is not a payout");
    }
    const [updatedTransaction] = await db.update(transactions).set({
      status: "completed",
      payoutDestination: payoutDetails.payoutDestination,
      payoutAccountNumber: payoutDetails.payoutAccountNumber,
      payoutAccountName: payoutDetails.payoutAccountName,
      payoutBankName: payoutDetails.payoutBankName,
      processedBy: payoutDetails.processedBy,
      processedAt: /* @__PURE__ */ new Date()
    }).where(eq2(transactions.id, id)).returning();
    if (transaction.status !== "completed") {
      const amount = parseFloat(transaction.amount);
      await db.update(members).set({
        totalPayouts: sql2`${members.totalPayouts} + ${amount}`,
        walletBalance: sql2`${members.walletBalance} - ${amount}`
      }).where(eq2(members.id, transaction.memberId));
    }
    if (transaction.planId) {
      await db.update(savingsPlans).set({ payoutStatus: "completed" }).where(eq2(savingsPlans.id, transaction.planId));
    }
    return updatedTransaction;
  }
  async getSavingsPlan(id) {
    const result = await db.query.savingsPlans.findFirst({
      where: eq2(savingsPlans.id, id),
      with: {
        member: true,
        contributions: {
          orderBy: [desc(planContributions.contributionNumber)]
        }
      }
    });
    return result;
  }
  async getAllSavingsPlans() {
    return await db.select().from(savingsPlans).orderBy(desc(savingsPlans.startDate));
  }
  async getPlansPaginated(page, limit) {
    const offset = (page - 1) * limit;
    const [data, countResult] = await Promise.all([
      db.select().from(savingsPlans).orderBy(desc(savingsPlans.startDate)).limit(limit).offset(offset),
      db.select({ count: sql2`count(*)` }).from(savingsPlans)
    ]);
    const total = Number(countResult[0]?.count) || 0;
    return {
      data,
      total,
      page,
      limit,
      totalPages: Math.ceil(total / limit)
    };
  }
  async getMemberSavingsPlans(memberId) {
    return await db.select().from(savingsPlans).where(eq2(savingsPlans.memberId, memberId)).orderBy(desc(savingsPlans.startDate));
  }
  async createSavingsPlan(insertPlan) {
    const [plan] = await db.insert(savingsPlans).values({
      ...insertPlan,
      maxContributions: insertPlan.maxContributions || 31,
      maxDays: insertPlan.maxDays || 62
    }).returning();
    return plan;
  }
  async createPlanContribution(insertContribution) {
    const plan = await db.query.savingsPlans.findFirst({
      where: eq2(savingsPlans.id, insertContribution.planId)
    });
    if (!plan) {
      throw new Error("Savings plan not found");
    }
    if (plan.status !== "active") {
      throw new Error("Cannot contribute to a closed savings plan");
    }
    const amount = parseFloat(insertContribution.amount);
    const contributionNumber = plan.contributionsCount + 1;
    const [contribution] = await db.insert(planContributions).values({
      ...insertContribution,
      contributionNumber
    }).returning();
    await db.update(savingsPlans).set({
      contributionsCount: contributionNumber,
      totalSaved: sql2`${savingsPlans.totalSaved} + ${amount}`
    }).where(eq2(savingsPlans.id, insertContribution.planId));
    await this.checkAndClosePlan(insertContribution.planId);
    return contribution;
  }
  async getPlanRemainingSlots(planId) {
    const plan = await db.query.savingsPlans.findFirst({
      where: eq2(savingsPlans.id, planId)
    });
    if (!plan) {
      throw new Error("Savings plan not found");
    }
    return Math.max(0, plan.maxContributions - plan.contributionsCount);
  }
  async createMultiplePlanContributions(planId, memberId, totalAmount, paymentMethod, notes, date, recordedBy) {
    const plan = await db.query.savingsPlans.findFirst({
      where: eq2(savingsPlans.id, planId)
    });
    if (!plan) {
      throw new Error("Savings plan not found");
    }
    if (plan.status !== "active") {
      throw new Error("Cannot contribute to a closed savings plan");
    }
    const dailyContribution = parseFloat(plan.contributionAmount);
    const remainingSlots = plan.maxContributions - plan.contributionsCount;
    const numContributions = Math.min(Math.floor(totalAmount / dailyContribution), remainingSlots);
    if (numContributions === 0 && remainingSlots > 0) {
      throw new Error(`Amount must be at least \u20A6${dailyContribution.toFixed(2)}`);
    }
    const contributions = [];
    let currentContributionNumber = plan.contributionsCount;
    const totalToSave = numContributions * dailyContribution;
    const overflowAmount = totalAmount - totalToSave;
    for (let i = 0; i < numContributions; i++) {
      currentContributionNumber += 1;
      const [contribution] = await db.insert(planContributions).values({
        planId,
        memberId,
        amount: dailyContribution.toString(),
        contributionNumber: currentContributionNumber,
        paymentMethod: paymentMethod || null,
        notes: notes ? `${notes} (contribution ${i + 1}/${numContributions})` : null,
        date: date ? new Date(date) : /* @__PURE__ */ new Date(),
        recordedBy: recordedBy || null
      }).returning();
      contributions.push(contribution);
    }
    if (numContributions > 0) {
      await db.update(savingsPlans).set({
        contributionsCount: currentContributionNumber,
        totalSaved: sql2`${savingsPlans.totalSaved} + ${totalToSave}`
      }).where(eq2(savingsPlans.id, planId));
      await this.checkAndClosePlan(planId);
    }
    const updatedPlan = await db.query.savingsPlans.findFirst({
      where: eq2(savingsPlans.id, planId)
    });
    const newRemainingSlots = updatedPlan ? updatedPlan.maxContributions - updatedPlan.contributionsCount : 0;
    return { contributions, overflowAmount, remainingSlots: newRemainingSlots };
  }
  async checkAndClosePlan(planId) {
    const plan = await db.query.savingsPlans.findFirst({
      where: eq2(savingsPlans.id, planId)
    });
    if (!plan || plan.status !== "active") {
      return;
    }
    if (plan.contributionsCount >= plan.maxContributions) {
      const totalSaved = parseFloat(plan.totalSaved);
      await db.update(savingsPlans).set({
        status: "completed",
        completedDate: /* @__PURE__ */ new Date(),
        payoutStatus: "pending"
      }).where(eq2(savingsPlans.id, planId));
      await this.addToWallet(plan.memberId, totalSaved);
      await this.createNotification({
        type: "plan_completed",
        title: "Savings Plan Completed",
        message: `Savings plan "${plan.planName}" has been completed with \u20A6${totalSaved.toFixed(2)} transferred to wallet. Payout is pending.`,
        memberId: plan.memberId,
        read: "false"
      });
    }
  }
  async checkAndCloseExpiredPlans() {
    const now = /* @__PURE__ */ new Date();
    const activePlans = await db.select().from(savingsPlans).where(eq2(savingsPlans.status, "active"));
    for (const plan of activePlans) {
      const startDate = new Date(plan.startDate);
      const daysSinceStart = Math.floor((now.getTime() - startDate.getTime()) / (1e3 * 60 * 60 * 24));
      if (daysSinceStart >= plan.maxDays) {
        const totalSaved = parseFloat(plan.totalSaved);
        await db.update(savingsPlans).set({
          status: "closed",
          completedDate: /* @__PURE__ */ new Date(),
          payoutStatus: "pending"
        }).where(eq2(savingsPlans.id, plan.id));
        if (totalSaved > 0) {
          await this.addToWallet(plan.memberId, totalSaved);
        }
        await this.createNotification({
          type: "plan_closed",
          title: "Savings Plan Auto-Closed",
          message: `Savings plan "${plan.planName}" has been automatically closed after ${plan.maxDays} days. \u20A6${totalSaved.toFixed(2)} transferred to wallet. Payout is pending.`,
          memberId: plan.memberId,
          read: "false"
        });
      }
    }
  }
  async requestPlanPayout(planId, memberId) {
    const plan = await db.query.savingsPlans.findFirst({
      where: eq2(savingsPlans.id, planId)
    });
    if (!plan) {
      throw new Error("Savings plan not found");
    }
    const member = await db.query.members.findFirst({
      where: eq2(members.id, memberId)
    });
    if (!member) {
      throw new Error("Member not found");
    }
    const totalSaved = parseFloat(plan.totalSaved);
    const [transaction] = await db.insert(transactions).values({
      memberId,
      planId,
      type: "payout",
      amount: totalSaved.toString(),
      status: "pending",
      notes: `Payout request for savings plan: ${plan.planName}`
    }).returning();
    await db.update(savingsPlans).set({ payoutStatus: "requested" }).where(eq2(savingsPlans.id, planId));
    await this.createNotification({
      type: "pending_payout",
      title: "Payout Requested",
      message: `Payout of \u20A6${totalSaved.toFixed(2)} requested for savings plan "${plan.planName}".`,
      memberId,
      transactionId: transaction.id,
      read: "false"
    });
    return transaction;
  }
  async completePlanPayout(transactionId, staffId, payoutDetails) {
    return this.completePayoutTransaction(transactionId, {
      ...payoutDetails,
      processedBy: staffId
    });
  }
  async addToWallet(memberId, amount) {
    const [member] = await db.update(members).set({
      walletBalance: sql2`${members.walletBalance} + ${amount}`
    }).where(eq2(members.id, memberId)).returning();
    return member;
  }
  async withdrawFromWallet(memberId, amount) {
    const member = await db.query.members.findFirst({
      where: eq2(members.id, memberId)
    });
    if (!member) {
      throw new Error("Member not found");
    }
    const walletBalance = parseFloat(member.walletBalance);
    if (amount > walletBalance) {
      throw new Error(`Insufficient wallet balance. Available: \u20A6${walletBalance.toFixed(2)}`);
    }
    const [updatedMember] = await db.update(members).set({
      walletBalance: sql2`${members.walletBalance} - ${amount}`
    }).where(eq2(members.id, memberId)).returning();
    return updatedMember;
  }
  async getDashboardStats() {
    const allMembers = await db.select().from(members);
    const activeMembers = allMembers.filter((m) => m.status === "active").length;
    const totalSavings = allMembers.reduce((sum, m) => sum + parseFloat(m.totalSavings), 0);
    const today = /* @__PURE__ */ new Date();
    today.setHours(0, 0, 0, 0);
    const tomorrow = new Date(today);
    tomorrow.setDate(tomorrow.getDate() + 1);
    const todayTransactions = await db.select().from(transactions).where(sql2`${transactions.date} >= ${today} AND ${transactions.date} < ${tomorrow} AND ${transactions.type} = 'savings'`);
    const todayCollections = todayTransactions.reduce((sum, t) => sum + parseFloat(t.amount), 0);
    const pendingPayoutTransactions = await db.select().from(transactions).where(sql2`${transactions.type} = 'payout' AND ${transactions.status} = 'pending'`);
    const pendingPayouts = pendingPayoutTransactions.reduce((sum, t) => sum + parseFloat(t.amount), 0);
    const [branchCount, staffCount] = await Promise.all([
      db.select({ count: sql2`count(*)` }).from(branches),
      db.select({ count: sql2`count(*)` }).from(staff)
    ]);
    return {
      totalSavings: totalSavings.toFixed(2),
      activeMembers,
      todayCollections: todayCollections.toFixed(2),
      pendingPayouts: pendingPayouts.toFixed(2),
      totalBranches: Number(branchCount[0]?.count) || 0,
      totalStaff: Number(staffCount[0]?.count) || 0
    };
  }
  async getAnalytics() {
    const last30Days = /* @__PURE__ */ new Date();
    last30Days.setDate(last30Days.getDate() - 30);
    const savingsTransactions = await db.select().from(transactions).where(sql2`${transactions.type} = 'savings' AND ${transactions.date} >= ${last30Days}`);
    const dailySavingsMap = /* @__PURE__ */ new Map();
    savingsTransactions.forEach((t) => {
      const dateStr = new Date(t.date).toISOString().split("T")[0];
      const existing = dailySavingsMap.get(dateStr) || { amount: 0, count: 0 };
      existing.amount += parseFloat(t.amount);
      existing.count += 1;
      dailySavingsMap.set(dateStr, existing);
    });
    const dailySavings = Array.from(dailySavingsMap.entries()).map(([date, data]) => ({ date, ...data })).sort((a, b) => new Date(a.date).getTime() - new Date(b.date).getTime());
    const allMembers = await db.select().from(members);
    const activeCount = allMembers.filter((m) => m.status === "active").length;
    const inactiveCount = allMembers.filter((m) => m.status === "inactive").length;
    const totalWalletBalance = allMembers.reduce((sum, m) => sum + parseFloat(m.walletBalance), 0);
    const allPlans = await db.select().from(savingsPlans);
    const activePlans = allPlans.filter((p) => p.status === "active").length;
    const completedPlans = allPlans.filter((p) => p.status === "completed" || p.status === "closed").length;
    const totalAmountLocked = allPlans.filter((p) => p.status === "active").reduce((sum, p) => sum + parseFloat(p.totalSaved), 0);
    const allTransactions = await db.select().from(transactions);
    const payoutTransactions = allTransactions.filter((t) => t.type === "payout");
    const completedPayouts = payoutTransactions.filter((t) => t.status === "completed").reduce((sum, t) => sum + parseFloat(t.amount), 0);
    const pendingPayouts = payoutTransactions.filter((t) => t.status === "pending").reduce((sum, t) => sum + parseFloat(t.amount), 0);
    const rejectedPayouts = payoutTransactions.filter((t) => t.status === "rejected").reduce((sum, t) => sum + parseFloat(t.amount), 0);
    return {
      dailySavings,
      memberStats: {
        activeCount,
        inactiveCount,
        totalWalletBalance: totalWalletBalance.toFixed(2)
      },
      planStats: {
        activePlans,
        completedPlans,
        totalAmountLocked: totalAmountLocked.toFixed(2)
      },
      payoutStats: {
        completed: completedPayouts.toFixed(2),
        pending: pendingPayouts.toFixed(2),
        rejected: rejectedPayouts.toFixed(2)
      }
    };
  }
  async getAllNotifications() {
    return await db.select().from(notifications).orderBy(desc(notifications.createdAt));
  }
  async getNotificationsPaginated(page, limit) {
    const offset = (page - 1) * limit;
    const [data, countResult] = await Promise.all([
      db.select().from(notifications).orderBy(desc(notifications.createdAt)).limit(limit).offset(offset),
      db.select({ count: sql2`count(*)` }).from(notifications)
    ]);
    const total = Number(countResult[0]?.count) || 0;
    return {
      data,
      total,
      page,
      limit,
      totalPages: Math.ceil(total / limit)
    };
  }
  async createNotification(notification) {
    const [newNotification] = await db.insert(notifications).values(notification).returning();
    return newNotification;
  }
  async markNotificationAsRead(id) {
    const [notif] = await db.update(notifications).set({ read: "true" }).where(eq2(notifications.id, id)).returning();
    return notif;
  }
  async getUnreadNotificationCount() {
    const result = await db.select().from(notifications).where(eq2(notifications.read, "false"));
    return result.length;
  }
  async getLoan(id) {
    const result = await db.query.loans.findFirst({
      where: eq2(loans.id, id),
      with: {
        member: true,
        plan: true,
        approvedByStaff: true
      }
    });
    return result;
  }
  async getAllLoans() {
    const result = await db.query.loans.findMany({
      with: {
        member: true,
        plan: true,
        approvedByStaff: true
      },
      orderBy: [desc(loans.requestedAt)]
    });
    return result;
  }
  async getLoansPaginated(page, limit) {
    const offset = (page - 1) * limit;
    const [data, countResult] = await Promise.all([
      db.query.loans.findMany({
        with: { member: true, plan: true, approvedByStaff: true },
        orderBy: [desc(loans.requestedAt)],
        limit,
        offset
      }),
      db.select({ count: sql2`count(*)` }).from(loans)
    ]);
    const total = Number(countResult[0]?.count) || 0;
    return {
      data,
      total,
      page,
      limit,
      totalPages: Math.ceil(total / limit)
    };
  }
  async getMemberLoans(memberId) {
    return await db.select().from(loans).where(eq2(loans.memberId, memberId)).orderBy(desc(loans.requestedAt));
  }
  async createLoan(insertLoan) {
    const plan = await db.query.savingsPlans.findFirst({
      where: eq2(savingsPlans.id, insertLoan.planId)
    });
    if (!plan) {
      throw new Error("Savings plan not found");
    }
    if (plan.status !== "active") {
      throw new Error("Can only borrow from an active savings plan");
    }
    const amount = parseFloat(insertLoan.amount);
    const totalSaved = parseFloat(plan.totalSaved);
    const maxBorrowable = totalSaved * 0.5;
    if (amount > maxBorrowable) {
      throw new Error(`Maximum borrowable amount is \u20A6${maxBorrowable.toFixed(2)} (50% of saved amount)`);
    }
    const existingLoan = await db.select().from(loans).where(and2(
      eq2(loans.planId, insertLoan.planId),
      eq2(loans.status, "pending")
    )).limit(1);
    if (existingLoan.length > 0) {
      throw new Error("There is already a pending loan request for this savings plan");
    }
    const activeLoan = await db.select().from(loans).where(and2(
      eq2(loans.planId, insertLoan.planId),
      eq2(loans.status, "active")
    )).limit(1);
    if (activeLoan.length > 0) {
      throw new Error("There is already an active loan on this savings plan. Please repay it first.");
    }
    const interestRate = 5;
    const totalRepayment = amount + amount * interestRate / 100;
    const dueDate = /* @__PURE__ */ new Date();
    dueDate.setDate(dueDate.getDate() + 30);
    const [loan] = await db.insert(loans).values({
      memberId: insertLoan.memberId,
      planId: insertLoan.planId,
      amount: amount.toString(),
      interestRate: interestRate.toString(),
      totalRepayment: totalRepayment.toString(),
      dueDate,
      notes: insertLoan.notes
    }).returning();
    return loan;
  }
  async approveLoan(loanId, staffId) {
    const loan = await db.query.loans.findFirst({
      where: eq2(loans.id, loanId)
    });
    if (!loan) {
      throw new Error("Loan not found");
    }
    if (loan.status !== "pending") {
      throw new Error("Loan is not pending approval");
    }
    const [updatedLoan] = await db.update(loans).set({
      status: "active",
      approvedAt: /* @__PURE__ */ new Date(),
      approvedBy: staffId
    }).where(eq2(loans.id, loanId)).returning();
    await this.addToWallet(loan.memberId, parseFloat(loan.amount));
    const member = await db.query.members.findFirst({
      where: eq2(members.id, loan.memberId)
    });
    await this.createNotification({
      type: "loan_approved",
      title: "Loan Approved",
      message: `Your loan of \u20A6${parseFloat(loan.amount).toFixed(2)} has been approved and credited to your wallet.`,
      memberId: loan.memberId,
      read: "false"
    });
    return updatedLoan;
  }
  async rejectLoan(loanId, staffId) {
    const loan = await db.query.loans.findFirst({
      where: eq2(loans.id, loanId)
    });
    if (!loan) {
      throw new Error("Loan not found");
    }
    if (loan.status !== "pending") {
      throw new Error("Loan is not pending approval");
    }
    const [updatedLoan] = await db.update(loans).set({
      status: "rejected"
    }).where(eq2(loans.id, loanId)).returning();
    await this.createNotification({
      type: "loan_rejected",
      title: "Loan Rejected",
      message: `Your loan request of \u20A6${parseFloat(loan.amount).toFixed(2)} has been rejected.`,
      memberId: loan.memberId,
      read: "false"
    });
    return updatedLoan;
  }
  async repayLoan(loanId, amount) {
    const loan = await db.query.loans.findFirst({
      where: eq2(loans.id, loanId)
    });
    if (!loan) {
      throw new Error("Loan not found");
    }
    if (loan.status !== "active") {
      throw new Error("Loan is not active");
    }
    const member = await db.query.members.findFirst({
      where: eq2(members.id, loan.memberId)
    });
    if (!member) {
      throw new Error("Member not found");
    }
    const walletBalance = parseFloat(member.walletBalance);
    if (amount > walletBalance) {
      throw new Error(`Insufficient wallet balance. Available: \u20A6${walletBalance.toFixed(2)}`);
    }
    const currentPaid = parseFloat(loan.amountPaid);
    const totalRepayment = parseFloat(loan.totalRepayment);
    const newAmountPaid = currentPaid + amount;
    let newStatus = loan.status;
    if (newAmountPaid >= totalRepayment) {
      newStatus = "paid";
    }
    const [updatedLoan] = await db.update(loans).set({
      amountPaid: newAmountPaid.toString(),
      status: newStatus
    }).where(eq2(loans.id, loanId)).returning();
    await this.withdrawFromWallet(loan.memberId, amount);
    if (newStatus === "paid") {
      await this.createNotification({
        type: "loan_approved",
        title: "Loan Fully Repaid",
        message: `Your loan has been fully repaid. Thank you!`,
        memberId: loan.memberId,
        read: "false"
      });
    }
    return updatedLoan;
  }
  async getMembersByBranch(branchId) {
    const branchStaff = await db.select().from(staff).where(eq2(staff.branchId, branchId));
    const staffIds = branchStaff.map((s) => s.id);
    if (staffIds.length === 0) {
      return [];
    }
    const allMembers = [];
    for (const staffId of staffIds) {
      const membersList = await db.select().from(members).where(eq2(members.staffId, staffId));
      allMembers.push(...membersList);
    }
    return allMembers;
  }
  async getMembersPaginatedByBranch(branchId, page, limit) {
    const branchMembers = await this.getMembersByBranch(branchId);
    const total = branchMembers.length;
    const offset = (page - 1) * limit;
    const data = branchMembers.slice(offset, offset + limit);
    return {
      data,
      total,
      page,
      limit,
      totalPages: Math.ceil(total / limit)
    };
  }
  async isMemberInBranch(memberId, branchId) {
    const member = await db.query.members.findFirst({
      where: eq2(members.id, memberId),
      with: {
        staff: true
      }
    });
    if (!member || !member.staff) {
      return false;
    }
    return member.staff.branchId === branchId;
  }
  async isStaffInBranch(staffId, branchId) {
    const staffMember = await db.query.staff.findFirst({
      where: eq2(staff.id, staffId)
    });
    if (!staffMember) {
      return false;
    }
    return staffMember.branchId === branchId;
  }
  async getInvestmentType(id) {
    const result = await db.query.investmentTypes.findFirst({
      where: eq2(investmentTypes.id, id),
      with: {
        createdByStaff: true
      }
    });
    return result;
  }
  async getAllInvestmentTypes() {
    return await db.select().from(investmentTypes).orderBy(desc(investmentTypes.createdAt));
  }
  async getActiveInvestmentTypes() {
    return await db.select().from(investmentTypes).where(eq2(investmentTypes.status, "active")).orderBy(desc(investmentTypes.createdAt));
  }
  async getInvestmentTypesPaginated(page, limit) {
    const offset = (page - 1) * limit;
    const [data, countResult] = await Promise.all([
      db.select().from(investmentTypes).orderBy(desc(investmentTypes.createdAt)).limit(limit).offset(offset),
      db.select({ count: sql2`count(*)` }).from(investmentTypes)
    ]);
    const total = Number(countResult[0]?.count) || 0;
    return {
      data,
      total,
      page,
      limit,
      totalPages: Math.ceil(total / limit)
    };
  }
  async createInvestmentType(insertData) {
    const [investmentType] = await db.insert(investmentTypes).values(insertData).returning();
    return investmentType;
  }
  async updateInvestmentType(id, updateData) {
    const [investmentType] = await db.update(investmentTypes).set({ ...updateData, updatedAt: /* @__PURE__ */ new Date() }).where(eq2(investmentTypes.id, id)).returning();
    return investmentType;
  }
  async deleteInvestmentType(id) {
    await db.delete(investmentTypes).where(eq2(investmentTypes.id, id));
  }
  async getMemberInvestment(id) {
    const result = await db.query.memberInvestments.findFirst({
      where: eq2(memberInvestments.id, id),
      with: {
        member: true,
        investmentType: true,
        assignedByStaff: true
      }
    });
    return result;
  }
  async getAllMemberInvestments() {
    const result = await db.query.memberInvestments.findMany({
      with: {
        member: true,
        investmentType: true,
        assignedByStaff: true
      },
      orderBy: [desc(memberInvestments.createdAt)]
    });
    return result;
  }
  async getMemberInvestmentsPaginated(page, limit, memberId, investmentTypeId) {
    const offset = (page - 1) * limit;
    let whereClause;
    if (memberId && investmentTypeId) {
      whereClause = and2(eq2(memberInvestments.memberId, memberId), eq2(memberInvestments.investmentTypeId, investmentTypeId));
    } else if (memberId) {
      whereClause = eq2(memberInvestments.memberId, memberId);
    } else if (investmentTypeId) {
      whereClause = eq2(memberInvestments.investmentTypeId, investmentTypeId);
    }
    const dataQuery = db.query.memberInvestments.findMany({
      where: whereClause,
      with: { member: true, investmentType: true, assignedByStaff: true },
      orderBy: [desc(memberInvestments.createdAt)],
      limit,
      offset
    });
    const countQuery = whereClause ? db.select({ count: sql2`count(*)` }).from(memberInvestments).where(whereClause) : db.select({ count: sql2`count(*)` }).from(memberInvestments);
    const [data, countResult] = await Promise.all([dataQuery, countQuery]);
    const total = Number(countResult[0]?.count) || 0;
    return {
      data,
      total,
      page,
      limit,
      totalPages: Math.ceil(total / limit)
    };
  }
  async getMemberInvestmentsByMember(memberId) {
    const result = await db.query.memberInvestments.findMany({
      where: eq2(memberInvestments.memberId, memberId),
      with: {
        member: true,
        investmentType: true,
        assignedByStaff: true
      },
      orderBy: [desc(memberInvestments.createdAt)]
    });
    return result;
  }
  async createMemberInvestment(insertData) {
    const investmentType = await db.query.investmentTypes.findFirst({
      where: eq2(investmentTypes.id, insertData.investmentTypeId)
    });
    if (!investmentType) {
      throw new Error("Investment type not found");
    }
    if (investmentType.status !== "active") {
      throw new Error("Investment type is not active");
    }
    const amount = parseFloat(insertData.amount);
    const minDeposit = parseFloat(investmentType.minimumDeposit);
    if (amount < minDeposit) {
      throw new Error(`Minimum deposit is \u20A6${minDeposit.toFixed(2)}`);
    }
    const interestRate = insertData.interestRate ? parseFloat(insertData.interestRate) : parseFloat(investmentType.interestRate);
    const expectedReturn = amount + amount * interestRate * investmentType.durationDays / 365 / 100;
    const maturityDate = /* @__PURE__ */ new Date();
    maturityDate.setDate(maturityDate.getDate() + investmentType.durationDays);
    const [investment] = await db.insert(memberInvestments).values({
      memberId: insertData.memberId,
      investmentTypeId: insertData.investmentTypeId,
      amount: amount.toString(),
      interestRate: interestRate.toString(),
      expectedReturn: expectedReturn.toFixed(2),
      maturityDate,
      notes: insertData.notes,
      assignedBy: insertData.assignedBy
    }).returning();
    return investment;
  }
  async breakMemberInvestment(id) {
    const investment = await db.query.memberInvestments.findFirst({
      where: eq2(memberInvestments.id, id),
      with: {
        investmentType: true
      }
    });
    if (!investment) {
      throw new Error("Investment not found");
    }
    if (investment.status !== "active") {
      throw new Error("Investment is not active");
    }
    const investmentType = investment.investmentType;
    if (!investmentType.isBreakable) {
      throw new Error("This investment cannot be broken");
    }
    const amount = parseFloat(investment.amount);
    const breakFeePercent = parseFloat(investmentType.breakFee);
    const breakFeeAmount = amount * breakFeePercent / 100;
    const actualReturn = amount - breakFeeAmount;
    const [updatedInvestment] = await db.update(memberInvestments).set({
      status: "broken",
      breakFeeApplied: breakFeeAmount.toFixed(2),
      actualReturn: actualReturn.toFixed(2),
      completedDate: /* @__PURE__ */ new Date()
    }).where(eq2(memberInvestments.id, id)).returning();
    await this.addToWallet(investment.memberId, actualReturn);
    return updatedInvestment;
  }
  async matureMemberInvestment(id) {
    const investment = await db.query.memberInvestments.findFirst({
      where: eq2(memberInvestments.id, id)
    });
    if (!investment) {
      throw new Error("Investment not found");
    }
    if (investment.status !== "active") {
      throw new Error("Investment is not active");
    }
    const expectedReturn = parseFloat(investment.expectedReturn);
    const [updatedInvestment] = await db.update(memberInvestments).set({
      status: "matured",
      actualReturn: investment.expectedReturn,
      completedDate: /* @__PURE__ */ new Date()
    }).where(eq2(memberInvestments.id, id)).returning();
    await this.addToWallet(investment.memberId, expectedReturn);
    return updatedInvestment;
  }
  async completeMemberInvestment(id) {
    const investment = await db.query.memberInvestments.findFirst({
      where: eq2(memberInvestments.id, id)
    });
    if (!investment) {
      throw new Error("Investment not found");
    }
    if (investment.status !== "matured") {
      throw new Error("Investment must be matured first");
    }
    const [updatedInvestment] = await db.update(memberInvestments).set({
      status: "completed"
    }).where(eq2(memberInvestments.id, id)).returning();
    return updatedInvestment;
  }
  async checkAndMatureInvestments() {
    const now = /* @__PURE__ */ new Date();
    const activeInvestments = await db.select().from(memberInvestments).where(and2(
      eq2(memberInvestments.status, "active"),
      lt(memberInvestments.maturityDate, now)
    ));
    for (const investment of activeInvestments) {
      try {
        await this.matureMemberInvestment(investment.id);
      } catch (error) {
        console.error(`Failed to mature investment ${investment.id}:`, error);
      }
    }
  }
  // Yearly Savings Plan Methods
  async getYearlySavingsPlan(id) {
    const result = await db.query.yearlySavingsPlans.findFirst({
      where: eq2(yearlySavingsPlans.id, id),
      with: {
        member: true,
        contributions: {
          orderBy: desc(yearlyPlanContributions.date)
        }
      }
    });
    return result;
  }
  async getAllYearlySavingsPlans() {
    return await db.query.yearlySavingsPlans.findMany({
      orderBy: desc(yearlySavingsPlans.createdAt)
    });
  }
  async getYearlyPlansPaginated(page, limit) {
    const offset = (page - 1) * limit;
    const [data, totalResult] = await Promise.all([
      db.query.yearlySavingsPlans.findMany({
        limit,
        offset,
        orderBy: desc(yearlySavingsPlans.createdAt),
        with: {
          member: true
        }
      }),
      db.select({ count: sql2`count(*)::int` }).from(yearlySavingsPlans)
    ]);
    const total = totalResult[0].count;
    return {
      data,
      total,
      page,
      limit,
      totalPages: Math.ceil(total / limit)
    };
  }
  async getMemberYearlySavingsPlans(memberId) {
    return await db.query.yearlySavingsPlans.findMany({
      where: eq2(yearlySavingsPlans.memberId, memberId),
      orderBy: desc(yearlySavingsPlans.createdAt)
    });
  }
  async createYearlySavingsPlan(plan) {
    const planData = {
      ...plan,
      maturityDate: new Date(plan.maturityDate)
    };
    const [newPlan] = await db.insert(yearlySavingsPlans).values(planData).returning();
    return newPlan;
  }
  async createYearlyPlanContribution(contribution) {
    const plan = await db.query.yearlySavingsPlans.findFirst({
      where: eq2(yearlySavingsPlans.id, contribution.planId)
    });
    if (!plan) {
      throw new Error("Yearly savings plan not found");
    }
    const contributionData = {
      planId: contribution.planId,
      memberId: contribution.memberId,
      amount: contribution.amount,
      contributionNumber: plan.contributionsCount + 1,
      date: contribution.date ? new Date(contribution.date) : /* @__PURE__ */ new Date(),
      paymentMethod: contribution.paymentMethod,
      notes: contribution.notes,
      recordedBy: contribution.recordedBy
    };
    const [newContribution] = await db.insert(yearlyPlanContributions).values(contributionData).returning();
    await db.update(yearlySavingsPlans).set({
      totalSaved: sql2`total_saved + ${contribution.amount}`,
      contributionsCount: sql2`contributions_count + 1`
    }).where(eq2(yearlySavingsPlans.id, contribution.planId));
    return newContribution;
  }
  async createMultipleYearlyPlanContributions(planId, memberId, totalAmount, paymentMethod, notes, date, recordedBy) {
    const plan = await this.getYearlySavingsPlan(planId);
    if (!plan) {
      throw new Error("Yearly savings plan not found");
    }
    if (plan.status !== "active") {
      throw new Error("Yearly savings plan is not active");
    }
    const contributionAmount = parseFloat(plan.contributionAmount);
    const maxContributions = plan.maxContributions - plan.contributionsCount;
    const contributionsToCreate = Math.min(Math.floor(totalAmount / contributionAmount), maxContributions);
    const overflowAmount = totalAmount - contributionsToCreate * contributionAmount;
    const contributions = [];
    const contributionDate = date ? new Date(date) : /* @__PURE__ */ new Date();
    for (let i = 0; i < contributionsToCreate; i++) {
      const contributionNumber = plan.contributionsCount + i + 1;
      const [contribution] = await db.insert(yearlyPlanContributions).values({
        planId,
        memberId,
        amount: contributionAmount.toString(),
        contributionNumber,
        date: contributionDate,
        paymentMethod,
        notes,
        recordedBy
      }).returning();
      contributions.push(contribution);
    }
    const totalAdded = contributionsToCreate * contributionAmount;
    await db.update(yearlySavingsPlans).set({
      totalSaved: sql2`total_saved + ${totalAdded}`,
      contributionsCount: sql2`contributions_count + ${contributionsToCreate}`
    }).where(eq2(yearlySavingsPlans.id, planId));
    if (plan.contributionsCount + contributionsToCreate >= plan.maxContributions) {
      await this.checkAndMatureYearlyPlan(planId);
    }
    return {
      contributions,
      overflowAmount,
      remainingSlots: Math.max(0, plan.maxContributions - plan.contributionsCount - contributionsToCreate)
    };
  }
  async checkAndMatureYearlyPlan(planId) {
    const plan = await db.query.yearlySavingsPlans.findFirst({
      where: eq2(yearlySavingsPlans.id, planId)
    });
    if (!plan) {
      throw new Error("Yearly savings plan not found");
    }
    if (plan.status !== "active") {
      return;
    }
    const isComplete = plan.contributionsCount >= plan.maxContributions;
    const isExpired = /* @__PURE__ */ new Date() > new Date(plan.maturityDate);
    if (isComplete || isExpired) {
      const profitCalculation = await this.calculateYearlyPlanProfit(planId);
      await db.update(yearlySavingsPlans).set({
        status: "matured",
        completedDate: /* @__PURE__ */ new Date(),
        profitEarned: profitCalculation.fullProfitEarned
        // Full profit at maturity
      }).where(eq2(yearlySavingsPlans.id, planId));
      const totalWithProfit = parseFloat(profitCalculation.fullProfitEarned) + parseFloat(plan.totalSaved);
      await this.addToWallet(plan.memberId, totalWithProfit);
      await this.createNotification({
        type: "yearly_plan_matured",
        title: "Yearly Savings Plan Matured",
        message: `Your yearly savings plan "${plan.planName}" has matured. \u20A6${totalWithProfit.toFixed(2)} (including \u20A6${profitCalculation.fullProfitEarned} profit) has been transferred to your wallet.`,
        memberId: plan.memberId,
        read: "false"
      });
    }
  }
  async checkAndMatureExpiredYearlyPlans() {
    const now = /* @__PURE__ */ new Date();
    const activePlans = await db.select().from(yearlySavingsPlans).where(and2(
      eq2(yearlySavingsPlans.status, "active"),
      lt(yearlySavingsPlans.maturityDate, now)
    ));
    for (const plan of activePlans) {
      try {
        await this.checkAndMatureYearlyPlan(plan.id);
      } catch (error) {
        console.error(`Failed to mature yearly plan ${plan.id}:`, error);
      }
    }
  }
  async getYearlyPlanRemainingSlots(planId) {
    const plan = await db.query.yearlySavingsPlans.findFirst({
      where: eq2(yearlySavingsPlans.id, planId)
    });
    if (!plan) {
      throw new Error("Yearly savings plan not found");
    }
    return Math.max(0, plan.maxContributions - plan.contributionsCount);
  }
  async requestYearlyPlanPayout(planId, memberId) {
    const plan = await this.getYearlySavingsPlan(planId);
    if (!plan) {
      throw new Error("Yearly savings plan not found");
    }
    if (plan.status !== "matured") {
      throw new Error("Yearly savings plan must be matured before payout");
    }
    const profitCalculation = await this.calculateYearlyPlanProfit(planId);
    const totalAmount = parseFloat(profitCalculation.totalWithProfit);
    const transaction = await this.createTransaction({
      memberId,
      planId,
      type: "payout",
      amount: totalAmount.toString(),
      status: "pending",
      date: (/* @__PURE__ */ new Date()).toISOString()
    });
    await db.update(yearlySavingsPlans).set({ payoutStatus: "pending" }).where(eq2(yearlySavingsPlans.id, planId));
    return transaction;
  }
  async completeYearlyPlanPayout(transactionId, staffId, payoutDetails) {
    const transaction = await this.getTransaction(transactionId);
    if (!transaction) {
      throw new Error("Transaction not found");
    }
    const updatedTransaction = await this.completePayoutTransaction(transactionId, {
      ...payoutDetails,
      processedBy: staffId
    });
    if (transaction.planId) {
      await db.update(yearlySavingsPlans).set({ payoutStatus: "completed" }).where(eq2(yearlySavingsPlans.id, transaction.planId));
    }
    return updatedTransaction;
  }
  async calculateYearlyPlanProfit(planId) {
    const plan = await db.query.yearlySavingsPlans.findFirst({
      where: eq2(yearlySavingsPlans.id, planId)
    });
    if (!plan) {
      throw new Error("Yearly savings plan not found");
    }
    const totalSaved = parseFloat(plan.totalSaved);
    const profitRate = parseFloat(plan.profitRate);
    const completedMonths = Math.floor(plan.contributionsCount / 31);
    const monthlyProfitRate = profitRate / 12;
    const halfMonthlyProfitRate = monthlyProfitRate / 2;
    const monthlyProfitEarned = totalSaved * halfMonthlyProfitRate * (completedMonths / 12);
    const fullProfitEarned = totalSaved * (profitRate / 100);
    const profitEarned = monthlyProfitEarned;
    const totalWithProfit = totalSaved + profitEarned;
    return {
      profitEarned: profitEarned.toFixed(2),
      totalWithProfit: totalWithProfit.toFixed(2),
      monthlyProfitEarned: monthlyProfitEarned.toFixed(2),
      fullProfitEarned: fullProfitEarned.toFixed(2)
    };
  }
  async requestYearlyPlanMonthlyPayout(planId, memberId) {
    const plan = await this.getYearlySavingsPlan(planId);
    if (!plan) {
      throw new Error("Yearly savings plan not found");
    }
    if (plan.status !== "active") {
      throw new Error("Yearly savings plan must be active before monthly payout");
    }
    const profitCalculation = await this.calculateYearlyPlanProfit(planId);
    const monthlyProfit = parseFloat(profitCalculation.monthlyProfitEarned);
    if (monthlyProfit <= 0) {
      throw new Error("No monthly profit available for payout");
    }
    const transaction = await this.createTransaction({
      memberId,
      planId,
      type: "monthly_payout",
      amount: monthlyProfit.toString(),
      status: "pending",
      date: (/* @__PURE__ */ new Date()).toISOString()
    });
    return transaction;
  }
  async checkAndPayMonthlyProfits() {
    const plans = await db.query.yearlySavingsPlans.findMany({
      where: {
        status: "active"
      }
    });
    for (const plan of plans) {
      const completedMonths = Math.floor(plan.contributionsCount / 31);
      if (completedMonths > 0) {
        const profitCalculation = await this.calculateYearlyPlanProfit(plan.id);
        const monthlyProfit = parseFloat(profitCalculation.monthlyProfitEarned);
        if (monthlyProfit > 0) {
          await this.addToWallet(plan.memberId, monthlyProfit);
          await this.createNotification({
            type: "monthly_payout",
            title: "Monthly Profit Paid",
            message: `Your yearly savings plan "${plan.planName}" has paid \u20A6${monthlyProfit.toFixed(2)} monthly profit (${completedMonths} months completed).`,
            memberId: plan.memberId,
            read: "false"
          });
        }
      }
    }
  }
  async getYearlySavingsPlan(planId) {
    const plan = await db.query.yearlySavingsPlans.findFirst({
      where: eq2(yearlySavingsPlans.id, planId)
    });
    return plan;
  }
  // Dynamic Savings Plan Type Management
  async getSavingsPlanTypes() {
    return await db.query.savingsPlanTypes.findMany({
      orderBy: [desc(savingsPlanTypes.createdAt)]
    });
  }
  async getSavingsPlanType(id) {
    const planType = await db.query.savingsPlanTypes.findFirst({
      where: eq2(savingsPlanTypes.id, id)
    });
    return planType;
  }
  async createSavingsPlanType(planType) {
    const [newPlanType] = await db.insert(savingsPlanTypes).values(planType).returning();
    return newPlanType;
  }
  async updateSavingsPlanType(id, planType) {
    const [updatedPlanType] = await db.update(savingsPlanTypes).set({ ...planType, updatedAt: /* @__PURE__ */ new Date() }).where(eq2(savingsPlanTypes.id, id)).returning();
    return updatedPlanType;
  }
  async deleteSavingsPlanType(id) {
    await db.delete(savingsPlanTypes).where(eq2(savingsPlanTypes.id, id));
  }
  // Dynamic Savings Plan Management
  async getDynamicSavingsPlans() {
    const plans = await db.query.dynamicSavingsPlans.findMany({
      with: {
        planType: true,
        member: true,
        contributions: true
      },
      orderBy: [desc(dynamicSavingsPlans.createdAt)]
    });
    return plans;
  }
  async getDynamicSavingsPlansPaginated(page, limit) {
    const offset = (page - 1) * limit;
    const plans = await db.query.dynamicSavingsPlans.findMany({
      with: {
        planType: true,
        member: true,
        contributions: true
      },
      orderBy: [desc(dynamicSavingsPlans.createdAt)],
      limit,
      offset
    });
    const totalCount = await db.select({ count: sql2`count(*)` }).from(dynamicSavingsPlans);
    const total = Number(totalCount[0]?.count || 0);
    return {
      data: plans,
      total,
      page,
      limit,
      totalPages: Math.ceil(total / limit)
    };
  }
  async getDynamicSavingsPlan(id) {
    const plan = await db.query.dynamicSavingsPlans.findFirst({
      with: {
        planType: true,
        member: true,
        contributions: true
      },
      where: eq2(dynamicSavingsPlans.id, id)
    });
    return plan;
  }
  async getMemberDynamicSavingsPlans(memberId) {
    const plans = await db.query.dynamicSavingsPlans.findMany({
      with: {
        planType: true,
        member: true,
        contributions: true
      },
      where: eq2(dynamicSavingsPlans.memberId, memberId),
      orderBy: [desc(dynamicSavingsPlans.createdAt)]
    });
    return plans;
  }
  async createDynamicSavingsPlan(plan) {
    const [newPlan] = await db.insert(dynamicSavingsPlans).values({
      ...plan,
      maturityDate: new Date(plan.maturityDate)
    }).returning();
    return newPlan;
  }
  async createDynamicSavingsPlanContribution(contribution) {
    const [newContribution] = await db.insert(dynamicSavingsPlanContributions).values({
      ...contribution,
      date: new Date(contribution.date || /* @__PURE__ */ new Date())
    }).returning();
    await db.update(dynamicSavingsPlans).set({
      currentContributions: sql2`current_contributions + 1`,
      totalSaved: sql2`total_saved + ${contribution.amount}`,
      updatedAt: /* @__PURE__ */ new Date()
    }).where(eq2(dynamicSavingsPlans.id, contribution.planId));
    return newContribution;
  }
  async createMultipleDynamicSavingsPlanContributions(planId, memberId, totalAmount, paymentMethod, notes, date, recordedBy) {
    const plan = await this.getDynamicSavingsPlan(planId);
    if (!plan) {
      throw new Error("Dynamic savings plan not found");
    }
    const contributionAmount = parseFloat(plan.contributionAmount);
    const count = Math.floor(totalAmount / contributionAmount);
    const remainingAmount = totalAmount % contributionAmount;
    const contributions = [];
    const currentDate = new Date(date || /* @__PURE__ */ new Date());
    for (let i = 0; i < count; i++) {
      const contributionDate = new Date(currentDate);
      contributionDate.setDate(currentDate.getDate() + i);
      const contribution = await this.createDynamicSavingsPlanContribution({
        planId,
        memberId,
        amount: contributionAmount.toString(),
        date: contributionDate.toISOString(),
        paymentMethod,
        notes: notes ? `${notes} - ${i + 1}/${count}` : notes,
        recordedBy
      });
      contributions.push(contribution);
    }
    return {
      count,
      contributions,
      remainingAmount: remainingAmount.toString()
    };
  }
  async calculateDynamicPlanProfit(planId) {
    const plan = await this.getDynamicSavingsPlan(planId);
    if (!plan) {
      throw new Error("Dynamic savings plan not found");
    }
    const totalSaved = parseFloat(plan.totalSaved);
    const interestRate = parseFloat(plan.interestRate);
    const breakFee = parseFloat(plan.breakFee);
    const earlyWithdrawalPenalty = parseFloat(plan.earlyWithdrawalPenalty);
    let completedPeriods = 0;
    switch (plan.profitCalculationType) {
      case "monthly":
        completedPeriods = Math.floor(plan.currentContributions / 31);
        break;
      case "quarterly":
        completedPeriods = Math.floor(plan.currentContributions / 93);
        break;
      case "biannually":
        completedPeriods = Math.floor(plan.currentContributions / 186);
        break;
      case "yearly":
        completedPeriods = Math.floor(plan.currentContributions / 372);
        break;
    }
    const periodProfitRate = interestRate / 12;
    const halfPeriodProfitRate = periodProfitRate / 2;
    const periodProfitEarned = totalSaved * halfPeriodProfitRate * (completedPeriods / 12);
    const fullProfitEarned = totalSaved * (interestRate / 100);
    const breakFeeAmount = totalSaved * (breakFee / 100);
    const earlyWithdrawalPenaltyAmount = totalSaved * (earlyWithdrawalPenalty / 100);
    const profitEarned = periodProfitEarned;
    const totalWithProfit = totalSaved + profitEarned;
    return {
      profitEarned: profitEarned.toFixed(2),
      totalWithProfit: totalWithProfit.toFixed(2),
      monthlyProfitEarned: periodProfitEarned.toFixed(2),
      fullProfitEarned: fullProfitEarned.toFixed(2),
      breakFeeAmount: breakFeeAmount.toFixed(2),
      earlyWithdrawalPenaltyAmount: earlyWithdrawalPenaltyAmount.toFixed(2)
    };
  }
  async checkAndMatureDynamicPlan(planId) {
    const plan = await this.getDynamicSavingsPlan(planId);
    if (!plan) {
      throw new Error("Dynamic savings plan not found");
    }
    if (plan.status !== "active") {
      return;
    }
    const isComplete = plan.currentContributions >= plan.maxContributions;
    const isExpired = /* @__PURE__ */ new Date() > new Date(plan.maturityDate);
    if (isComplete || isExpired) {
      const profitCalculation = await this.calculateDynamicPlanProfit(planId);
      await db.update(dynamicSavingsPlans).set({
        status: "matured",
        completedDate: /* @__PURE__ */ new Date(),
        profitEarned: profitCalculation.fullProfitEarned,
        totalWithProfit: (parseFloat(plan.totalSaved) + parseFloat(profitCalculation.fullProfitEarned)).toString()
      }).where(eq2(dynamicSavingsPlans.id, planId));
      const totalWithProfit = parseFloat(profitCalculation.fullProfitEarned) + parseFloat(plan.totalSaved);
      await this.addToWallet(plan.memberId, totalWithProfit);
      await this.createNotification({
        type: "plan_completed",
        title: "Dynamic Savings Plan Matured",
        message: `Your dynamic savings plan "${plan.planName}" has matured. \u20A6${totalWithProfit.toFixed(2)} (including \u20A6${profitCalculation.fullProfitEarned} profit) has been transferred to your wallet.`,
        memberId: plan.memberId,
        read: "false"
      });
    }
  }
  async checkAndMatureExpiredDynamicPlans() {
    const now = /* @__PURE__ */ new Date();
    const activePlans = await db.select().from(dynamicSavingsPlans).where(eq2(dynamicSavingsPlans.status, "active"));
    for (const plan of activePlans) {
      if (new Date(plan.maturityDate) <= now) {
        await this.checkAndMatureDynamicPlan(plan.id);
      }
    }
  }
  async getDynamicPlanRemainingSlots(planId) {
    const plan = await this.getDynamicSavingsPlan(planId);
    if (!plan) {
      throw new Error("Dynamic savings plan not found");
    }
    return Math.max(0, plan.maxContributions - plan.currentContributions);
  }
  async requestDynamicPlanPayout(planId, memberId) {
    const plan = await this.getDynamicSavingsPlan(planId);
    if (!plan) {
      throw new Error("Dynamic savings plan not found");
    }
    if (plan.status !== "matured") {
      throw new Error("Dynamic savings plan must be matured before payout");
    }
    const profitCalculation = await this.calculateDynamicPlanProfit(planId);
    const totalWithProfit = parseFloat(profitCalculation.totalWithProfit);
    const transaction = await this.createTransaction({
      memberId,
      planId: planId.toString(),
      type: "payout",
      amount: totalWithProfit.toString(),
      status: "pending",
      date: (/* @__PURE__ */ new Date()).toISOString()
    });
    await db.update(dynamicSavingsPlans).set({ payoutStatus: "pending" }).where(eq2(dynamicSavingsPlans.id, planId));
    return transaction;
  }
  async completeDynamicPlanPayout(transactionId, staffId, payoutDetails) {
    const transaction = await this.getTransaction(transactionId);
    if (!transaction) {
      throw new Error("Transaction not found");
    }
    const updatedTransaction = await this.completePayoutTransaction(transactionId, staffId, {
      payoutDestination: payoutDetails.payoutDestination,
      payoutAccountNumber: payoutDetails.payoutAccountNumber,
      payoutAccountName: payoutDetails.payoutAccountName,
      payoutBankName: payoutDetails.payoutBankName,
      processedBy: staffId
    });
    if (transaction.planId) {
      await db.update(dynamicSavingsPlans).set({ payoutStatus: "completed" }).where(eq2(dynamicSavingsPlans.id, parseInt(transaction.planId)));
    }
    return updatedTransaction;
  }
  async breakDynamicPlan(planId, memberId, reason) {
    const plan = await this.getDynamicSavingsPlan(planId);
    if (!plan) {
      throw new Error("Dynamic savings plan not found");
    }
    if (plan.status !== "active") {
      throw new Error("Dynamic savings plan must be active to break");
    }
    const daysSinceStart = Math.floor(((/* @__PURE__ */ new Date()).getTime() - new Date(plan.startDate).getTime()) / (1e3 * 60 * 60 * 24));
    if (daysSinceStart < plan.canBreakAfterDays) {
      throw new Error(`Plan can only be broken after ${plan.canBreakAfterDays} days`);
    }
    const profitCalculation = await this.calculateDynamicPlanProfit(planId);
    const breakFeeAmount = parseFloat(profitCalculation.breakFeeAmount);
    const payoutAmount = parseFloat(plan.totalSaved) - breakFeeAmount;
    await db.update(dynamicSavingsPlans).set({
      status: "broken",
      completedDate: /* @__PURE__ */ new Date(),
      profitEarned: "0",
      // No profit when breaking
      totalWithProfit: payoutAmount.toString()
    }).where(eq2(dynamicSavingsPlans.id, planId));
    const transaction = await this.createTransaction({
      memberId,
      planId: planId.toString(),
      type: "payout",
      amount: payoutAmount.toString(),
      status: "completed",
      date: (/* @__PURE__ */ new Date()).toISOString(),
      notes: reason || `Plan broken with break fee of \u20A6${breakFeeAmount.toFixed(2)}`
    });
    await this.addToWallet(memberId, payoutAmount);
    await this.createNotification({
      type: "plan_completed",
      title: "Dynamic Savings Plan Broken",
      message: `Your dynamic savings plan "${plan.planName}" has been broken. \u20A6${payoutAmount.toFixed(2)} has been transferred to your wallet after break fee of \u20A6${breakFeeAmount.toFixed(2)}.`,
      memberId: plan.memberId,
      read: "false"
    });
    return transaction;
  }
  async requestDynamicPlanEarlyWithdrawal(planId, memberId) {
    const plan = await this.getDynamicSavingsPlan(planId);
    if (!plan) {
      throw new Error("Dynamic savings plan not found");
    }
    if (plan.status !== "active") {
      throw new Error("Dynamic savings plan must be active for early withdrawal");
    }
    const profitCalculation = await this.calculateDynamicPlanProfit(planId);
    const earlyWithdrawalPenaltyAmount = parseFloat(profitCalculation.earlyWithdrawalPenaltyAmount);
    const payoutAmount = parseFloat(plan.totalSaved) - earlyWithdrawalPenaltyAmount;
    await db.update(dynamicSavingsPlans).set({
      status: "broken",
      completedDate: /* @__PURE__ */ new Date(),
      profitEarned: "0",
      // No profit for early withdrawal
      totalWithProfit: payoutAmount.toString()
    }).where(eq2(dynamicSavingsPlans.id, planId));
    const transaction = await this.createTransaction({
      memberId,
      planId: planId.toString(),
      type: "payout",
      amount: payoutAmount.toString(),
      status: "pending",
      date: (/* @__PURE__ */ new Date()).toISOString(),
      notes: `Early withdrawal with penalty of \u20A6${earlyWithdrawalPenaltyAmount.toFixed(2)}`
    });
    return transaction;
  }
};
var storage = new DatabaseStorage();

// server/routes.ts
import { fromZodError } from "zod-validation-error";
import PDFDocument from "pdfkit";
import * as XLSX from "xlsx";
async function registerRoutes(app2) {
  setupAuth(app2);
  setInterval(async () => {
    try {
      await storage.checkAndCloseExpiredPlans();
      await storage.checkAndMatureExpiredYearlyPlans();
    } catch (error) {
      console.error("Error checking expired plans:", error);
    }
  }, 60 * 60 * 1e3);
  app2.get("/api/dashboard/stats", requireAuth, async (req, res) => {
    try {
      const stats = await storage.getDashboardStats();
      res.json(stats);
    } catch (error) {
      res.status(500).json({ message: error.message || "Failed to fetch dashboard stats" });
    }
  });
  app2.get("/api/analytics", requireAuth, async (req, res) => {
    try {
      const analytics = await storage.getAnalytics();
      res.json(analytics);
    } catch (error) {
      res.status(500).json({ message: error.message || "Failed to fetch analytics" });
    }
  });
  app2.get("/api/branches", requireAuth, async (req, res) => {
    try {
      const page = Math.max(1, parseInt(req.query.page) || 1);
      const limit = Math.min(100, Math.max(1, parseInt(req.query.limit) || 10));
      const paginated = await storage.getBranchesPaginated(page, limit);
      res.json(paginated);
    } catch (error) {
      res.status(500).json({ message: error.message || "Failed to fetch branches" });
    }
  });
  app2.get("/api/branches/all", requireAuth, async (req, res) => {
    try {
      const branches2 = await storage.getAllBranches();
      res.json(branches2);
    } catch (error) {
      res.status(500).json({ message: error.message || "Failed to fetch branches" });
    }
  });
  app2.get("/api/branches/:id", requireAuth, async (req, res) => {
    try {
      const branch = await storage.getBranch(req.params.id);
      if (!branch) {
        return res.status(404).json({ message: "Branch not found" });
      }
      res.json(branch);
    } catch (error) {
      res.status(500).json({ message: error.message || "Failed to fetch branch" });
    }
  });
  app2.post("/api/branches", requireRole("admin", "manager"), async (req, res) => {
    try {
      const result = insertBranchSchema.safeParse(req.body);
      if (!result.success) {
        return res.status(400).json({ message: fromZodError(result.error).message });
      }
      const branch = await storage.createBranch(result.data);
      res.status(201).json(branch);
    } catch (error) {
      res.status(500).json({ message: error.message || "Failed to create branch" });
    }
  });
  app2.put("/api/branches/:id", requireRole("admin", "manager"), async (req, res) => {
    try {
      const result = insertBranchSchema.partial().safeParse(req.body);
      if (!result.success) {
        return res.status(400).json({ message: fromZodError(result.error).message });
      }
      const branch = await storage.updateBranch(req.params.id, result.data);
      res.json(branch);
    } catch (error) {
      res.status(500).json({ message: error.message || "Failed to update branch" });
    }
  });
  app2.delete("/api/branches/:id", requireRole("admin"), async (req, res) => {
    try {
      await storage.deleteBranch(req.params.id);
      res.status(204).send();
    } catch (error) {
      res.status(500).json({ message: error.message || "Failed to delete branch" });
    }
  });
  app2.get("/api/staff", requireAuth, async (req, res) => {
    try {
      const page = Math.max(1, parseInt(req.query.page) || 1);
      const limit = Math.min(100, Math.max(1, parseInt(req.query.limit) || 10));
      const branchId = filterByUserBranch(req);
      if (branchId) {
        const branchStaff = await storage.getStaffByBranch(branchId);
        const total = branchStaff.length;
        const offset = (page - 1) * limit;
        const data = branchStaff.slice(offset, offset + limit);
        res.json({
          data,
          total,
          page,
          limit,
          totalPages: Math.ceil(total / limit)
        });
      } else {
        const paginated = await storage.getStaffPaginated(page, limit);
        res.json(paginated);
      }
    } catch (error) {
      res.status(500).json({ message: error.message || "Failed to fetch staff" });
    }
  });
  app2.get("/api/staff/all", requireAuth, async (req, res) => {
    try {
      const allStaff = await storage.getAllStaff();
      res.json(allStaff);
    } catch (error) {
      res.status(500).json({ message: error.message || "Failed to fetch staff" });
    }
  });
  app2.get("/api/staff/:id", requireAuth, async (req, res) => {
    try {
      const staffMember = await storage.getStaff(req.params.id);
      if (!staffMember) {
        return res.status(404).json({ message: "Staff not found" });
      }
      const branchId = filterByUserBranch(req);
      if (branchId) {
        const hasAccess = await storage.isStaffInBranch(req.params.id, branchId);
        if (!hasAccess) {
          return res.status(403).json({ message: "Forbidden: You can only access staff in your branch" });
        }
      }
      res.json(staffMember);
    } catch (error) {
      res.status(500).json({ message: error.message || "Failed to fetch staff" });
    }
  });
  app2.get("/api/branches/:branchId/staff", requireAuth, async (req, res) => {
    try {
      const staffList = await storage.getStaffByBranch(req.params.branchId);
      res.json(staffList);
    } catch (error) {
      res.status(500).json({ message: error.message || "Failed to fetch branch staff" });
    }
  });
  app2.post("/api/staff", requireRole("admin", "manager"), async (req, res) => {
    try {
      const result = insertStaffSchema.safeParse(req.body);
      if (!result.success) {
        return res.status(400).json({ message: fromZodError(result.error).message });
      }
      const staffMember = await storage.createStaff(result.data);
      res.status(201).json(staffMember);
    } catch (error) {
      res.status(500).json({ message: error.message || "Failed to create staff" });
    }
  });
  app2.put("/api/staff/:id", requireRole("admin", "manager"), async (req, res) => {
    try {
      const branchId = filterByUserBranch(req);
      if (branchId) {
        const hasAccess = await storage.isStaffInBranch(req.params.id, branchId);
        if (!hasAccess) {
          return res.status(403).json({ message: "Forbidden: You can only update staff in your branch" });
        }
      }
      const result = insertStaffSchema.partial().safeParse(req.body);
      if (!result.success) {
        return res.status(400).json({ message: fromZodError(result.error).message });
      }
      const staffMember = await storage.updateStaff(req.params.id, result.data);
      res.json(staffMember);
    } catch (error) {
      res.status(500).json({ message: error.message || "Failed to update staff" });
    }
  });
  app2.delete("/api/staff/:id", requireRole("admin"), async (req, res) => {
    try {
      const branchId = filterByUserBranch(req);
      if (branchId) {
        const hasAccess = await storage.isStaffInBranch(req.params.id, branchId);
        if (!hasAccess) {
          return res.status(403).json({ message: "Forbidden: You can only delete staff in your branch" });
        }
      }
      await storage.deleteStaff(req.params.id);
      res.status(204).send();
    } catch (error) {
      res.status(500).json({ message: error.message || "Failed to delete staff" });
    }
  });
  app2.get("/api/staff/:staffId/members", requireAuth, async (req, res) => {
    try {
      const members2 = await storage.getMembersByStaff(req.params.staffId);
      res.json(members2);
    } catch (error) {
      res.status(500).json({ message: error.message || "Failed to fetch staff members" });
    }
  });
  app2.get("/api/members", requireAuth, async (req, res) => {
    try {
      const page = Math.max(1, parseInt(req.query.page) || 1);
      const limit = Math.min(100, Math.max(1, parseInt(req.query.limit) || 10));
      const branchId = filterByUserBranch(req);
      if (branchId) {
        const paginated = await storage.getMembersPaginatedByBranch(branchId, page, limit);
        res.json(paginated);
      } else {
        const paginated = await storage.getMembersPaginated(page, limit);
        res.json(paginated);
      }
    } catch (error) {
      res.status(500).json({ message: error.message || "Failed to fetch members" });
    }
  });
  app2.get("/api/members/:id", requireAuth, async (req, res) => {
    try {
      const member = await storage.getMember(req.params.id);
      if (!member) {
        return res.status(404).json({ message: "Member not found" });
      }
      const branchId = filterByUserBranch(req);
      if (branchId) {
        const hasAccess = await storage.isMemberInBranch(req.params.id, branchId);
        if (!hasAccess) {
          return res.status(403).json({ message: "Forbidden: You can only access members in your branch" });
        }
      }
      res.json(member);
    } catch (error) {
      res.status(500).json({ message: error.message || "Failed to fetch member" });
    }
  });
  app2.get("/api/members/:id/with-staff", requireAuth, async (req, res) => {
    try {
      const member = await storage.getMemberWithStaff(req.params.id);
      if (!member) {
        return res.status(404).json({ message: "Member not found" });
      }
      res.json(member);
    } catch (error) {
      res.status(500).json({ message: error.message || "Failed to fetch member" });
    }
  });
  app2.post("/api/members", requireAuth, async (req, res) => {
    try {
      const result = insertMemberSchema.safeParse(req.body);
      if (!result.success) {
        return res.status(400).json({ message: fromZodError(result.error).message });
      }
      const member = await storage.createMember(result.data);
      res.status(201).json(member);
    } catch (error) {
      res.status(500).json({ message: error.message || "Failed to create member" });
    }
  });
  app2.put("/api/members/:id", requireAuth, async (req, res) => {
    try {
      const branchId = filterByUserBranch(req);
      if (branchId) {
        const hasAccess = await storage.isMemberInBranch(req.params.id, branchId);
        if (!hasAccess) {
          return res.status(403).json({ message: "Forbidden: You can only update members in your branch" });
        }
      }
      const result = insertMemberSchema.partial().safeParse(req.body);
      if (!result.success) {
        return res.status(400).json({ message: fromZodError(result.error).message });
      }
      const member = await storage.updateMember(req.params.id, result.data);
      res.json(member);
    } catch (error) {
      res.status(500).json({ message: error.message || "Failed to update member" });
    }
  });
  app2.delete("/api/members/:id", requireRole("admin", "manager"), async (req, res) => {
    try {
      const branchId = filterByUserBranch(req);
      if (branchId) {
        const hasAccess = await storage.isMemberInBranch(req.params.id, branchId);
        if (!hasAccess) {
          return res.status(403).json({ message: "Forbidden: You can only delete members in your branch" });
        }
      }
      await storage.deleteMember(req.params.id);
      res.status(204).send();
    } catch (error) {
      res.status(500).json({ message: error.message || "Failed to delete member" });
    }
  });
  app2.get("/api/transactions", requireAuth, async (req, res) => {
    try {
      const page = Math.max(1, parseInt(req.query.page) || 1);
      const limit = Math.min(100, Math.max(1, parseInt(req.query.limit) || 10));
      const paginated = await storage.getTransactionsPaginated(page, limit);
      res.json(paginated);
    } catch (error) {
      res.status(500).json({ message: error.message || "Failed to fetch transactions" });
    }
  });
  app2.get("/api/transactions/recent", requireAuth, async (req, res) => {
    try {
      const transactions2 = await storage.getRecentTransactions(10);
      res.json(transactions2);
    } catch (error) {
      res.status(500).json({ message: error.message || "Failed to fetch recent transactions" });
    }
  });
  app2.get("/api/transactions/payouts", requireAuth, async (req, res) => {
    try {
      const page = Math.max(1, parseInt(req.query.page) || 1);
      const limit = Math.min(100, Math.max(1, parseInt(req.query.limit) || 10));
      const paginated = await storage.getPayoutsPaginated(page, limit);
      res.json(paginated);
    } catch (error) {
      res.status(500).json({ message: error.message || "Failed to fetch payouts" });
    }
  });
  app2.post("/api/transactions", requireAuth, async (req, res) => {
    try {
      const result = insertTransactionSchema.safeParse(req.body);
      if (!result.success) {
        return res.status(400).json({ message: fromZodError(result.error).message });
      }
      const member = await storage.getMember(result.data.memberId);
      if (!member) {
        return res.status(404).json({ message: "Member not found" });
      }
      if (result.data.type === "payout") {
        const requestedAmount = parseFloat(result.data.amount);
        const walletBalance = parseFloat(member.walletBalance);
        if (result.data.status === "completed" && requestedAmount > walletBalance) {
          return res.status(400).json({
            message: `Insufficient wallet balance. Member has \u20A6${walletBalance.toFixed(2)} but requested \u20A6${requestedAmount.toFixed(2)}`
          });
        }
      }
      const transaction = await storage.createTransaction(result.data);
      res.status(201).json(transaction);
    } catch (error) {
      res.status(500).json({ message: error.message || "Failed to create transaction" });
    }
  });
  app2.patch("/api/transactions/:id/approve", requireRole("admin", "manager"), async (req, res) => {
    try {
      const transaction = await storage.getTransaction(req.params.id);
      if (!transaction) {
        return res.status(404).json({ message: "Transaction not found" });
      }
      if (transaction.type !== "payout") {
        return res.status(400).json({ message: "Only payout transactions can be approved" });
      }
      if (transaction.status !== "pending") {
        return res.status(400).json({ message: "Transaction is not pending" });
      }
      const member = await storage.getMember(transaction.memberId);
      if (!member) {
        return res.status(404).json({ message: "Member not found" });
      }
      const requestedAmount = parseFloat(transaction.amount);
      const walletBalance = parseFloat(member.walletBalance);
      if (requestedAmount > walletBalance) {
        return res.status(400).json({
          message: `Insufficient wallet balance. Member has \u20A6${walletBalance.toFixed(2)} but payout is \u20A6${requestedAmount.toFixed(2)}`
        });
      }
      const updatedTransaction = await storage.updateTransactionStatus(req.params.id, "approved");
      await storage.createNotification({
        type: "payout_approved",
        title: "Payout Approved",
        message: `Payout of \u20A6${requestedAmount.toFixed(2)} for ${member.name} has been approved. Awaiting processing.`,
        memberId: transaction.memberId,
        transactionId: transaction.id,
        read: "false"
      });
      res.json(updatedTransaction);
    } catch (error) {
      res.status(500).json({ message: error.message || "Failed to approve transaction" });
    }
  });
  app2.patch("/api/transactions/:id/complete-payout", requireAuth, async (req, res) => {
    try {
      const { payoutDestination, payoutAccountNumber, payoutAccountName, payoutBankName } = req.body;
      if (!payoutDestination || !payoutAccountNumber || !payoutAccountName || !payoutBankName) {
        return res.status(400).json({ message: "All payout details are required" });
      }
      const transaction = await storage.getTransaction(req.params.id);
      if (!transaction) {
        return res.status(404).json({ message: "Transaction not found" });
      }
      if (transaction.type !== "payout") {
        return res.status(400).json({ message: "Transaction is not a payout" });
      }
      if (transaction.status !== "pending" && transaction.status !== "approved") {
        return res.status(400).json({ message: "Transaction cannot be completed" });
      }
      const member = await storage.getMember(transaction.memberId);
      if (!member) {
        return res.status(404).json({ message: "Member not found" });
      }
      const requestedAmount = parseFloat(transaction.amount);
      const walletBalance = parseFloat(member.walletBalance);
      if (requestedAmount > walletBalance) {
        return res.status(400).json({
          message: `Insufficient wallet balance. Member has \u20A6${walletBalance.toFixed(2)} but payout is \u20A6${requestedAmount.toFixed(2)}`
        });
      }
      const updatedTransaction = await storage.completePayoutTransaction(req.params.id, {
        payoutDestination,
        payoutAccountNumber,
        payoutAccountName,
        payoutBankName,
        processedBy: req.user.id
      });
      await storage.createNotification({
        type: "payout_approved",
        title: "Payout Completed",
        message: `Payout of \u20A6${requestedAmount.toFixed(2)} for ${member.name} has been sent to ${payoutBankName} - ${payoutAccountNumber}.`,
        memberId: transaction.memberId,
        transactionId: transaction.id,
        read: "false"
      });
      res.json(updatedTransaction);
    } catch (error) {
      res.status(500).json({ message: error.message || "Failed to complete payout" });
    }
  });
  app2.patch("/api/transactions/:id/reject", requireRole("admin", "manager"), async (req, res) => {
    try {
      const transaction = await storage.getTransaction(req.params.id);
      if (!transaction) {
        return res.status(404).json({ message: "Transaction not found" });
      }
      if (transaction.type !== "payout") {
        return res.status(400).json({ message: "Only payout transactions can be rejected" });
      }
      if (transaction.status !== "pending" && transaction.status !== "approved") {
        return res.status(400).json({ message: "Transaction cannot be rejected" });
      }
      const member = await storage.getMember(transaction.memberId);
      const requestedAmount = parseFloat(transaction.amount);
      const updatedTransaction = await storage.updateTransactionStatus(req.params.id, "rejected");
      if (member) {
        await storage.createNotification({
          type: "pending_payout",
          title: "Payout Request Rejected",
          message: `Payout request of \u20A6${requestedAmount.toFixed(2)} for ${member.name} has been rejected.`,
          memberId: transaction.memberId,
          transactionId: transaction.id,
          read: "false"
        });
      }
      res.json(updatedTransaction);
    } catch (error) {
      res.status(500).json({ message: error.message || "Failed to reject transaction" });
    }
  });
  app2.get("/api/savings-plans", requireAuth, async (req, res) => {
    try {
      const page = Math.max(1, parseInt(req.query.page) || 1);
      const limit = Math.min(100, Math.max(1, parseInt(req.query.limit) || 10));
      const paginated = await storage.getPlansPaginated(page, limit);
      res.json(paginated);
    } catch (error) {
      res.status(500).json({ message: error.message || "Failed to fetch savings plans" });
    }
  });
  app2.get("/api/savings-plans/:id", requireAuth, async (req, res) => {
    try {
      const plan = await storage.getSavingsPlan(req.params.id);
      if (!plan) {
        return res.status(404).json({ message: "Savings plan not found" });
      }
      res.json(plan);
    } catch (error) {
      res.status(500).json({ message: error.message || "Failed to fetch savings plan" });
    }
  });
  app2.get("/api/savings-plans/:id/remaining-slots", requireAuth, async (req, res) => {
    try {
      const remainingSlots = await storage.getPlanRemainingSlots(req.params.id);
      res.json({ remainingSlots });
    } catch (error) {
      res.status(500).json({ message: error.message || "Failed to fetch remaining slots" });
    }
  });
  app2.get("/api/members/:memberId/plans", requireAuth, async (req, res) => {
    try {
      const plans = await storage.getMemberSavingsPlans(req.params.memberId);
      res.json(plans);
    } catch (error) {
      res.status(500).json({ message: error.message || "Failed to fetch member plans" });
    }
  });
  app2.post("/api/savings-plans", requireAuth, async (req, res) => {
    try {
      const result = insertSavingsPlanSchema.safeParse(req.body);
      if (!result.success) {
        return res.status(400).json({ message: fromZodError(result.error).message });
      }
      const plan = await storage.createSavingsPlan(result.data);
      res.status(201).json(plan);
    } catch (error) {
      res.status(500).json({ message: error.message || "Failed to create savings plan" });
    }
  });
  app2.post("/api/plans/:planId/contribute", requireAuth, async (req, res) => {
    try {
      const { amount, paymentMethod, notes, date, memberId } = req.body;
      if (!amount || isNaN(parseFloat(amount)) || parseFloat(amount) <= 0) {
        return res.status(400).json({ message: "Invalid amount" });
      }
      const result = await storage.createMultiplePlanContributions(
        req.params.planId,
        memberId,
        parseFloat(amount),
        paymentMethod,
        notes,
        date,
        req.user?.id
      );
      const plan = await storage.getSavingsPlan(req.params.planId);
      if (plan && (plan.status === "completed" || plan.status === "closed")) {
        const member = await storage.getMember(plan.memberId);
        if (member) {
          await storage.createNotification({
            type: "plan_completed",
            title: "Savings Plan Completed",
            message: `Savings plan "${plan.planName}" has been completed with \u20A6${parseFloat(plan.totalSaved).toFixed(2)} transferred to wallet.`,
            memberId: plan.memberId,
            read: "false"
          });
        }
      }
      res.status(201).json(result);
    } catch (error) {
      res.status(500).json({ message: error.message || "Failed to create contribution" });
    }
  });
  app2.post("/api/plans/:planId/request-payout", requireAuth, async (req, res) => {
    try {
      const plan = await storage.getSavingsPlan(req.params.planId);
      if (!plan) {
        return res.status(404).json({ message: "Savings plan not found" });
      }
      const transaction = await storage.requestPlanPayout(req.params.planId, plan.memberId);
      res.status(201).json(transaction);
    } catch (error) {
      res.status(500).json({ message: error.message || "Failed to request payout" });
    }
  });
  app2.post("/api/wallet/:memberId/withdraw", requireAuth, async (req, res) => {
    try {
      const { amount } = req.body;
      if (!amount || isNaN(parseFloat(amount)) || parseFloat(amount) <= 0) {
        return res.status(400).json({ message: "Invalid withdrawal amount" });
      }
      const member = await storage.withdrawFromWallet(req.params.memberId, parseFloat(amount));
      res.json(member);
    } catch (error) {
      res.status(500).json({ message: error.message || "Failed to process withdrawal" });
    }
  });
  app2.get("/api/yearly-savings-plans", requireAuth, async (req, res) => {
    try {
      const page = Math.max(1, parseInt(req.query.page) || 1);
      const limit = Math.min(100, Math.max(1, parseInt(req.query.limit) || 10));
      const paginated = await storage.getYearlyPlansPaginated(page, limit);
      res.json(paginated);
    } catch (error) {
      res.status(500).json({ message: error.message || "Failed to fetch yearly savings plans" });
    }
  });
  app2.get("/api/yearly-savings-plans/:id", requireAuth, async (req, res) => {
    try {
      const plan = await storage.getYearlySavingsPlan(req.params.id);
      if (!plan) {
        return res.status(404).json({ message: "Yearly savings plan not found" });
      }
      res.json(plan);
    } catch (error) {
      res.status(500).json({ message: error.message || "Failed to fetch yearly savings plan" });
    }
  });
  app2.get("/api/yearly-savings-plans/:id/remaining-slots", requireAuth, async (req, res) => {
    try {
      const remainingSlots = await storage.getYearlyPlanRemainingSlots(req.params.id);
      res.json({ remainingSlots });
    } catch (error) {
      res.status(500).json({ message: error.message || "Failed to fetch remaining slots" });
    }
  });
  app2.get("/api/yearly-savings-plans/:id/profit-calculation", requireAuth, async (req, res) => {
    try {
      const profitCalculation = await storage.calculateYearlyPlanProfit(req.params.id);
      res.json(profitCalculation);
    } catch (error) {
      res.status(500).json({ message: error.message || "Failed to calculate profit" });
    }
  });
  app2.get("/api/members/:memberId/yearly-plans", requireAuth, async (req, res) => {
    try {
      const plans = await storage.getMemberYearlySavingsPlans(req.params.memberId);
      res.json(plans);
    } catch (error) {
      res.status(500).json({ message: error.message || "Failed to fetch member yearly plans" });
    }
  });
  app2.post("/api/yearly-savings-plans", requireAuth, async (req, res) => {
    try {
      const result = insertYearlySavingsPlanSchema.safeParse(req.body);
      if (!result.success) {
        return res.status(400).json({ message: fromZodError(result.error).message });
      }
      const plan = await storage.createYearlySavingsPlan(result.data);
      res.status(201).json(plan);
    } catch (error) {
      res.status(500).json({ message: error.message || "Failed to create yearly savings plan" });
    }
  });
  app2.post("/api/yearly-plans/:planId/contribute", requireAuth, async (req, res) => {
    try {
      const { amount, paymentMethod, notes, date, memberId } = req.body;
      if (!amount || isNaN(parseFloat(amount)) || parseFloat(amount) <= 0) {
        return res.status(400).json({ message: "Invalid amount" });
      }
      const result = await storage.createMultipleYearlyPlanContributions(
        req.params.planId,
        memberId,
        parseFloat(amount),
        paymentMethod,
        notes,
        date,
        req.user?.id
      );
      const plan = await storage.getYearlySavingsPlan(req.params.planId);
      if (plan && plan.status === "matured") {
        const member = await storage.getMember(plan.memberId);
        if (member) {
          await storage.createNotification({
            type: "yearly_plan_matured",
            title: "Yearly Savings Plan Matured",
            message: `Your yearly savings plan "${plan.planName}" has matured with \u20A6${parseFloat(plan.totalSaved).toFixed(2)} saved and \u20A6${parseFloat(plan.profitEarned || "0").toFixed(2)} profit transferred to wallet.`,
            memberId: plan.memberId,
            read: "false"
          });
        }
      }
      res.status(201).json(result);
    } catch (error) {
      res.status(500).json({ message: error.message || "Failed to create contribution" });
    }
  });
  app2.post("/api/yearly-plans/:planId/request-payout", requireAuth, async (req, res) => {
    try {
      const plan = await storage.getYearlySavingsPlan(req.params.planId);
      if (!plan) {
        return res.status(404).json({ message: "Yearly savings plan not found" });
      }
      const transaction = await storage.requestYearlyPlanPayout(req.params.planId, plan.memberId);
      res.status(201).json(transaction);
    } catch (error) {
      res.status(500).json({ message: error.message || "Failed to request payout" });
    }
  });
  app2.post("/api/yearly-plans/:planId/request-monthly-payout", requireAuth, async (req, res) => {
    try {
      const plan = await storage.getYearlySavingsPlan(req.params.planId);
      if (!plan) {
        return res.status(404).json({ message: "Yearly savings plan not found" });
      }
      const transaction = await storage.requestYearlyPlanMonthlyPayout(req.params.planId, plan.memberId);
      res.status(201).json(transaction);
    } catch (error) {
      res.status(500).json({ message: error.message || "Failed to request monthly payout" });
    }
  });
  app2.post("/admin/yearly-plans/pay-monthly-profits", requireAuth, async (req, res) => {
    try {
      await storage.checkAndPayMonthlyProfits();
      res.json({ message: "Monthly profits paid successfully" });
    } catch (error) {
      res.status(500).json({ message: error.message || "Failed to pay monthly profits" });
    }
  });
  app2.get("/api/transactions/:id/receipt", requireAuth, async (req, res) => {
    try {
      const transaction = await storage.getTransaction(req.params.id);
      if (!transaction) {
        return res.status(404).json({ message: "Transaction not found" });
      }
      const doc = new PDFDocument();
      res.setHeader("Content-Type", "application/pdf");
      res.setHeader("Content-Disposition", `attachment; filename="receipt-${req.params.id}.pdf"`);
      doc.pipe(res);
      doc.fontSize(24).font("Helvetica-Bold").text("ApexL Investment", { align: "center" });
      doc.fontSize(10).font("Helvetica").text("Transaction Receipt", { align: "center" });
      doc.moveTo(50, doc.y).lineTo(550, doc.y).stroke();
      doc.moveDown();
      doc.fontSize(12).font("Helvetica-Bold").text("Receipt Information", { underline: true });
      doc.fontSize(10).font("Helvetica");
      doc.text(`Receipt #: ${req.params.id.substring(0, 8).toUpperCase()}`);
      doc.text(`Date: ${new Date(transaction.date).toLocaleDateString()}`);
      doc.text(`Time: ${new Date(transaction.createdAt).toLocaleTimeString()}`);
      doc.moveDown();
      doc.fontSize(12).font("Helvetica-Bold").text("Member Information", { underline: true });
      doc.fontSize(10).font("Helvetica");
      doc.text(`Name: ${transaction.member?.name || "N/A"}`);
      doc.text(`Email: ${transaction.member?.email || "N/A"}`);
      doc.text(`Phone: ${transaction.member?.phone || "N/A"}`);
      doc.text(`Wallet Number: ${transaction.member?.walletNumber || "N/A"}`);
      doc.moveDown();
      doc.fontSize(12).font("Helvetica-Bold").text("Transaction Details", { underline: true });
      doc.fontSize(10).font("Helvetica");
      doc.text(`Type: ${transaction.type.toUpperCase()}`);
      doc.text(`Amount: \u20A6${parseFloat(transaction.amount).toFixed(2)}`);
      doc.text(`Status: ${transaction.status.toUpperCase()}`);
      doc.text(`Payment Method: ${transaction.paymentMethod || "N/A"}`);
      if (transaction.notes) {
        doc.text(`Notes: ${transaction.notes}`);
      }
      doc.moveDown();
      doc.fontSize(8).text("This receipt confirms the transaction recorded in the ApexL Investment system.", { align: "center" });
      doc.text("For inquiries, contact admin@apexl.com", { align: "center" });
      doc.end();
    } catch (error) {
      res.status(500).json({ message: error.message || "Failed to generate receipt" });
    }
  });
  app2.get("/api/notifications", requireAuth, async (req, res) => {
    try {
      const page = Math.max(1, parseInt(req.query.page) || 1);
      const limit = Math.min(100, Math.max(1, parseInt(req.query.limit) || 10));
      const paginated = await storage.getNotificationsPaginated(page, limit);
      res.json(paginated);
    } catch (error) {
      res.status(500).json({ message: error.message || "Failed to fetch notifications" });
    }
  });
  app2.get("/api/notifications/unread/count", requireAuth, async (req, res) => {
    try {
      const count = await storage.getUnreadNotificationCount();
      res.json({ count });
    } catch (error) {
      res.status(500).json({ message: error.message || "Failed to fetch unread count" });
    }
  });
  app2.post("/api/notifications", requireAuth, async (req, res) => {
    try {
      const result = insertNotificationSchema.safeParse(req.body);
      if (!result.success) {
        return res.status(400).json({ message: fromZodError(result.error).message });
      }
      const notif = await storage.createNotification(result.data);
      res.status(201).json(notif);
    } catch (error) {
      res.status(500).json({ message: error.message || "Failed to create notification" });
    }
  });
  app2.patch("/api/notifications/:id/read", requireAuth, async (req, res) => {
    try {
      const notif = await storage.markNotificationAsRead(req.params.id);
      res.json(notif);
    } catch (error) {
      res.status(500).json({ message: error.message || "Failed to mark notification as read" });
    }
  });
  app2.get("/api/transactions/export/excel", requireAuth, async (req, res) => {
    try {
      const transactions2 = await storage.getAllTransactions();
      const data = transactions2.map((t) => ({
        Date: new Date(t.date).toLocaleDateString(),
        Member: t.member?.name || "Unknown",
        Type: t.type,
        Amount: parseFloat(t.amount).toFixed(2),
        "Payment Method": t.paymentMethod || "N/A",
        Status: t.status,
        Notes: t.notes || ""
      }));
      const workbook = XLSX.utils.book_new();
      const worksheet = XLSX.utils.json_to_sheet(data);
      XLSX.utils.book_append_sheet(workbook, worksheet, "Transactions");
      const buffer = XLSX.write(workbook, { bookType: "xlsx", type: "buffer" });
      res.setHeader("Content-Type", "application/vnd.openxmlformats-officedocument.spreadsheetml.sheet");
      res.setHeader("Content-Disposition", `attachment; filename="transactions-${(/* @__PURE__ */ new Date()).toISOString().split("T")[0]}.xlsx"`);
      res.send(buffer);
    } catch (error) {
      res.status(500).json({ message: error.message || "Failed to export transactions" });
    }
  });
  app2.get("/api/loans", requireAuth, async (req, res) => {
    try {
      const page = Math.max(1, parseInt(req.query.page) || 1);
      const limit = Math.min(100, Math.max(1, parseInt(req.query.limit) || 10));
      const paginated = await storage.getLoansPaginated(page, limit);
      res.json(paginated);
    } catch (error) {
      res.status(500).json({ message: error.message || "Failed to fetch loans" });
    }
  });
  app2.get("/api/loans/:id", requireAuth, async (req, res) => {
    try {
      const loan = await storage.getLoan(req.params.id);
      if (!loan) {
        return res.status(404).json({ message: "Loan not found" });
      }
      res.json(loan);
    } catch (error) {
      res.status(500).json({ message: error.message || "Failed to fetch loan" });
    }
  });
  app2.get("/api/members/:memberId/loans", requireAuth, async (req, res) => {
    try {
      const loans2 = await storage.getMemberLoans(req.params.memberId);
      res.json(loans2);
    } catch (error) {
      res.status(500).json({ message: error.message || "Failed to fetch member loans" });
    }
  });
  app2.post("/api/loans", requireAuth, async (req, res) => {
    try {
      const result = insertLoanSchema.safeParse(req.body);
      if (!result.success) {
        return res.status(400).json({ message: fromZodError(result.error).message });
      }
      const branchId = filterByUserBranch(req);
      if (branchId) {
        const hasAccess = await storage.isMemberInBranch(result.data.memberId, branchId);
        if (!hasAccess) {
          return res.status(403).json({ message: "Forbidden: You can only create loans for members in your branch" });
        }
      }
      const loan = await storage.createLoan(result.data);
      const member = await storage.getMember(loan.memberId);
      if (member) {
        await storage.createNotification({
          type: "loan_requested",
          title: "Loan Request Submitted",
          message: `Loan request of \u20A6${parseFloat(loan.amount).toFixed(2)} submitted by ${member.name}. Awaiting approval.`,
          memberId: loan.memberId,
          read: "false"
        });
      }
      res.status(201).json(loan);
    } catch (error) {
      res.status(500).json({ message: error.message || "Failed to create loan request" });
    }
  });
  app2.patch("/api/loans/:id/approve", requireRole("admin", "manager"), async (req, res) => {
    try {
      const loan = await storage.approveLoan(req.params.id, req.user.id);
      res.json(loan);
    } catch (error) {
      res.status(500).json({ message: error.message || "Failed to approve loan" });
    }
  });
  app2.patch("/api/loans/:id/reject", requireRole("admin", "manager"), async (req, res) => {
    try {
      const loan = await storage.rejectLoan(req.params.id, req.user.id);
      res.json(loan);
    } catch (error) {
      res.status(500).json({ message: error.message || "Failed to reject loan" });
    }
  });
  app2.post("/api/loans/:id/repay", requireAuth, async (req, res) => {
    try {
      const { amount } = req.body;
      if (!amount || isNaN(parseFloat(amount)) || parseFloat(amount) <= 0) {
        return res.status(400).json({ message: "Invalid repayment amount" });
      }
      const loan = await storage.repayLoan(req.params.id, parseFloat(amount));
      res.json(loan);
    } catch (error) {
      res.status(500).json({ message: error.message || "Failed to process loan repayment" });
    }
  });
  app2.get("/api/investment-types", requireAuth, async (req, res) => {
    try {
      const page = Math.max(1, parseInt(req.query.page) || 1);
      const limit = Math.min(100, Math.max(1, parseInt(req.query.limit) || 10));
      const paginated = await storage.getInvestmentTypesPaginated(page, limit);
      res.json(paginated);
    } catch (error) {
      res.status(500).json({ message: error.message || "Failed to fetch investment types" });
    }
  });
  app2.get("/api/investment-types/active", requireAuth, async (req, res) => {
    try {
      const types = await storage.getActiveInvestmentTypes();
      res.json(types);
    } catch (error) {
      res.status(500).json({ message: error.message || "Failed to fetch active investment types" });
    }
  });
  app2.get("/api/investment-types/:id", requireAuth, async (req, res) => {
    try {
      const investmentType = await storage.getInvestmentType(req.params.id);
      if (!investmentType) {
        return res.status(404).json({ message: "Investment type not found" });
      }
      res.json(investmentType);
    } catch (error) {
      res.status(500).json({ message: error.message || "Failed to fetch investment type" });
    }
  });
  app2.post("/api/investment-types", requireRole("admin"), async (req, res) => {
    try {
      const result = insertInvestmentTypeSchema.safeParse(req.body);
      if (!result.success) {
        return res.status(400).json({ message: fromZodError(result.error).message });
      }
      const investmentType = await storage.createInvestmentType({
        ...result.data,
        createdBy: req.user.id
      });
      res.status(201).json(investmentType);
    } catch (error) {
      res.status(500).json({ message: error.message || "Failed to create investment type" });
    }
  });
  app2.put("/api/investment-types/:id", requireRole("admin"), async (req, res) => {
    try {
      const result = insertInvestmentTypeSchema.partial().safeParse(req.body);
      if (!result.success) {
        return res.status(400).json({ message: fromZodError(result.error).message });
      }
      const investmentType = await storage.updateInvestmentType(req.params.id, result.data);
      res.json(investmentType);
    } catch (error) {
      res.status(500).json({ message: error.message || "Failed to update investment type" });
    }
  });
  app2.delete("/api/investment-types/:id", requireRole("admin"), async (req, res) => {
    try {
      await storage.deleteInvestmentType(req.params.id);
      res.status(204).send();
    } catch (error) {
      res.status(500).json({ message: error.message || "Failed to delete investment type" });
    }
  });
  app2.get("/api/investments", requireAuth, async (req, res) => {
    try {
      const page = Math.max(1, parseInt(req.query.page) || 1);
      const limit = Math.min(100, Math.max(1, parseInt(req.query.limit) || 10));
      const memberId = req.query.memberId;
      const investmentTypeId = req.query.investmentTypeId;
      const paginated = await storage.getMemberInvestmentsPaginated(page, limit, memberId, investmentTypeId);
      res.json(paginated);
    } catch (error) {
      res.status(500).json({ message: error.message || "Failed to fetch investments" });
    }
  });
  app2.get("/api/investments/:id", requireAuth, async (req, res) => {
    try {
      const investment = await storage.getMemberInvestment(req.params.id);
      if (!investment) {
        return res.status(404).json({ message: "Investment not found" });
      }
      res.json(investment);
    } catch (error) {
      res.status(500).json({ message: error.message || "Failed to fetch investment" });
    }
  });
  app2.get("/api/members/:memberId/investments", requireAuth, async (req, res) => {
    try {
      const investments = await storage.getMemberInvestmentsByMember(req.params.memberId);
      res.json(investments);
    } catch (error) {
      res.status(500).json({ message: error.message || "Failed to fetch member investments" });
    }
  });
  app2.post("/api/investments", requireAuth, async (req, res) => {
    try {
      const result = insertMemberInvestmentSchema.safeParse(req.body);
      if (!result.success) {
        return res.status(400).json({ message: fromZodError(result.error).message });
      }
      const branchId = filterByUserBranch(req);
      if (branchId) {
        const hasAccess = await storage.isMemberInBranch(result.data.memberId, branchId);
        if (!hasAccess) {
          return res.status(403).json({ message: "Forbidden: You can only create investments for members in your branch" });
        }
      }
      const investment = await storage.createMemberInvestment({
        ...result.data,
        assignedBy: req.user.id
      });
      const member = await storage.getMember(investment.memberId);
      if (member) {
        await storage.createNotification({
          type: "reminder",
          title: "New Investment Created",
          message: `Investment of \u20A6${parseFloat(investment.amount).toFixed(2)} has been created for ${member.name}. Expected return: \u20A6${parseFloat(investment.expectedReturn).toFixed(2)}.`,
          memberId: investment.memberId,
          read: "false"
        });
      }
      res.status(201).json(investment);
    } catch (error) {
      res.status(500).json({ message: error.message || "Failed to create investment" });
    }
  });
  app2.patch("/api/investments/:id/break", requireAuth, async (req, res) => {
    try {
      const investment = await storage.breakMemberInvestment(req.params.id);
      const member = await storage.getMember(investment.memberId);
      if (member) {
        await storage.createNotification({
          type: "reminder",
          title: "Investment Broken",
          message: `Investment has been broken early. Amount returned: \u20A6${parseFloat(investment.actualReturn || "0").toFixed(2)} (after break fee).`,
          memberId: investment.memberId,
          read: "false"
        });
      }
      res.json(investment);
    } catch (error) {
      res.status(500).json({ message: error.message || "Failed to break investment" });
    }
  });
  app2.patch("/api/investments/:id/mature", requireRole("admin", "manager"), async (req, res) => {
    try {
      const investment = await storage.matureMemberInvestment(req.params.id);
      const member = await storage.getMember(investment.memberId);
      if (member) {
        await storage.createNotification({
          type: "reminder",
          title: "Investment Matured",
          message: `Your investment has matured! Amount credited to wallet: \u20A6${parseFloat(investment.actualReturn || "0").toFixed(2)}.`,
          memberId: investment.memberId,
          read: "false"
        });
      }
      res.json(investment);
    } catch (error) {
      res.status(500).json({ message: error.message || "Failed to mature investment" });
    }
  });
  setInterval(async () => {
    try {
      await storage.checkAndMatureInvestments();
    } catch (error) {
      console.error("Error checking matured investments:", error);
    }
  }, 60 * 60 * 1e3);
  const httpServer = createServer(app2);
  return httpServer;
}

// server/app.ts
function log(message, source = "express") {
  const formattedTime = (/* @__PURE__ */ new Date()).toLocaleTimeString("en-US", {
    hour: "numeric",
    minute: "2-digit",
    second: "2-digit",
    hour12: true
  });
  console.log(`${formattedTime} [${source}] ${message}`);
}
var app = express();
app.use(express.json({
  verify: (req, _res, buf) => {
    req.rawBody = buf;
  }
}));
app.use(express.urlencoded({ extended: false }));
app.use((req, res, next) => {
  const start = Date.now();
  const path2 = req.path;
  let capturedJsonResponse = void 0;
  const originalResJson = res.json;
  res.json = function(bodyJson, ...args) {
    capturedJsonResponse = bodyJson;
    return originalResJson.apply(res, [bodyJson, ...args]);
  };
  res.on("finish", () => {
    const duration = Date.now() - start;
    if (path2.startsWith("/api")) {
      let logLine = `${req.method} ${path2} ${res.statusCode} in ${duration}ms`;
      if (capturedJsonResponse) {
        logLine += ` :: ${JSON.stringify(capturedJsonResponse)}`;
      }
      if (logLine.length > 80) {
        logLine = logLine.slice(0, 79) + "\u2026";
      }
      log(logLine);
    }
  });
  next();
});
async function runApp(setup) {
  const server = await registerRoutes(app);
  app.use((err, _req, res, _next) => {
    const status = err.status || err.statusCode || 500;
    const message = err.message || "Internal Server Error";
    res.status(status).json({ message });
    throw err;
  });
  await setup(app, server);
  const port = parseInt(process.env.PORT || "5000", 10);
  server.listen(port, "0.0.0.0", () => {
    log(`serving on port ${port}`);
  });
}

// server/index-prod.ts
async function serveStatic(app2, _server) {
  const distPath = path.resolve(import.meta.dirname, "public");
  if (!fs.existsSync(distPath)) {
    throw new Error(
      `Could not find the build directory: ${distPath}, make sure to build the client first`
    );
  }
  app2.use(express2.static(distPath));
  app2.use("*", (_req, res) => {
    res.sendFile(path.resolve(distPath, "index.html"));
  });
}
(async () => {
  await runApp(serveStatic);
})();
export {
  serveStatic
};
